//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "vtkCMBModelReader.h"

#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkCMBParserBase.h"
#include "vtkObjectFactory.h"
#include "vtkPolyData.h"
#include "vtkSmartPointer.h"
#include "vtkXMLPolyDataReader.h"
#include <sys/stat.h>

vtkStandardNewMacro(vtkCMBModelReader);

vtkCMBModelReader::vtkCMBModelReader()
{
  this->FileName = 0;
  this->SetNumberOfInputPorts(0);
}

vtkCMBModelReader:: ~vtkCMBModelReader()
{
  this->SetFileName(0);
}

int vtkCMBModelReader::RequestData(
  vtkInformation *vtkNotUsed(request),
  vtkInformationVector **vtkNotUsed(inputVector),
  vtkInformationVector *outputVector)
{
  vtkDebugMacro("Reading a CMB file.");
  // get the info object
  vtkInformation *outInfo = outputVector->GetInformationObject(0);

  // get the ouptut
  vtkPolyData *output = vtkPolyData::SafeDownCast(
    outInfo->Get(vtkDataObject::DATA_OBJECT()));

  struct stat fs;
  if ( stat(this->FileName, &fs) )
    {
    vtkErrorMacro(<< this->FileName << " not found");
    return 0;
    }

  vtkSmartPointer<vtkXMLPolyDataReader> reader =
    vtkSmartPointer<vtkXMLPolyDataReader>::New();
  reader->SetFileName(this->GetFileName());
  reader->Update();

  output->ShallowCopy(reader->GetOutput());

  return 1;
}

//----------------------------------------------------------------------------
int vtkCMBModelReader::RequestInformation(
  vtkInformation *vtkNotUsed(request),
  vtkInformationVector **vtkNotUsed(inputVector),
  vtkInformationVector *vtkNotUsed(outputVector))
{
  if (!this->FileName)
    {
    vtkErrorMacro("FileName has to be specified!");
    return 0;
    }

  return 1;
}

void vtkCMBModelReader::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "FileName: " << this->FileName << endl;
}
