/*****************************************************************************
*
* Copyright (c) 2000 - 2017, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                               avtVolumePlot.C                             //
// ************************************************************************* //

#include "avtVolumePlot.h"

#include <limits.h>
#include <math.h>

#include <avtCallback.h>
#include <avtCompactTreeFilter.h>
#include <avtDatabaseMetaData.h>
#include <avtGradientExpression.h>
#include <avtVolumeRenderer.h>
#include <avtLookupTable.h>
#include <avtResampleFilter.h>
#include <avtShiftCenteringFilter.h>
#include <avtUserDefinedMapper.h>
#include <avtVolumeFilter.h>
#include <avtLowerResolutionVolumeFilter.h>

#include <VolumeAttributes.h>

#include <DebugStream.h>
#include <ImproperUseException.h>
#include <LostConnectionException.h>

#include <string>

// ****************************************************************************
//  Method: avtVolumePlot constructor
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
//  Modifications:
//    Brad Whitlock, Thu Apr 19 15:10:55 PST 2001
//    Added a legend.
//
//    Kathleen Bonnell, Fri Aug 31 08:50:30 PDT 2001
//    Added avtLUT.
//
//    Hank Childs, Tue Nov 20 17:45:22 PST 2001
//    Initialized volume filter.
//
//    Hank Childs, Fri Feb  8 19:38:57 PST 2002
//    Initialized shift centering filter.
//
//    Hank Childs, Wed Apr 24 09:38:36 PDT 2002
//    Make use of instantiation method that insulate us from graphics libs.
//
//    Hank Childs, Mon May 20 10:47:18 PDT 2002
//    Added reference pointer construct because renderers are now reference
//    counted.
//
//    Eric Brugger, Wed Jul 16 11:35:47 PDT 2003
//    Modified to work with the new way legends are managed.
//
//    Hank Childs, Wed Nov 24 17:03:44 PST 2004
//    Removed hacks involved with previous software volume rendering mode.
//
//    Brad Whitlock, Mon Dec 15 15:51:38 PST 2008
//    I added another filter.
//
//    Brad Whitlock, Tue Jan 31 12:11:27 PST 2012
//    I added a compact tree filter.
//
// ****************************************************************************

avtVolumePlot::avtVolumePlot() : avtVolumeDataPlot()
{
    volumeFilter = NULL;
    volumeImageFilter = NULL;
    gradientFilter = NULL;
    resampleFilter = NULL;
    shiftCentering = NULL;
    compactTree = NULL;
    renderer = avtVolumeRenderer::New();

    avtCustomRenderer_p cr;
    CopyTo(cr, renderer);
    mapper = new avtUserDefinedMapper(cr);

    avtLUT = new avtLookupTable();

    varLegend = new avtVolumeVariableLegend;
    varLegend->SetTitle("Volume");

    //
    // This is to allow the legend to reference counted so the behavior can
    // still access it after the plot is deleted.  The legend cannot be
    // reference counted all of the time since we need to know that it is a
    // VariableLegend.
    //
    varLegendRefPtr = varLegend;
}

// ****************************************************************************
//  Method: avtVolumePlot destructor
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
//  Modifications:
//
//    Kathleen Bonnell, Fri Aug 31 08:50:30 PDT 2001A
//    Added avtLUT.
//
//    Hank Childs, Tue Nov 20 17:50:55 PST 2001
//    Delete volume filter.
//
//    Hank Childs, Fri Feb  8 19:38:57 PST 2002
//    Delete shift centering filter.
//
//    Hank Childs, Mon May 20 10:47:18 PDT 2002
//    Don't delete the renderer, since it is now reference counted.
//
//    Hank Childs, Wed Nov 24 17:03:44 PST 2004
//    Removed hacks involved with previous software volume rendering mode.
//
//    Brad Whitlock, Mon Dec 15 15:52:01 PST 2008
//    I added another filter.
//
//    Brad Whitlock, Tue Jan 31 12:11:27 PST 2012
//    I added a compact tree filter.
//
// ****************************************************************************

avtVolumePlot::~avtVolumePlot()
{
    delete mapper;
    delete shiftCentering;
    delete avtLUT;

    if (volumeFilter != NULL)
        delete volumeFilter;
    if (volumeImageFilter != NULL)
        delete volumeImageFilter;
    if (gradientFilter != NULL)
        delete gradientFilter;
    if (resampleFilter != NULL)
        delete resampleFilter;
    if (compactTree != NULL)
        delete compactTree;

    renderer = NULL;

    //
    // Do not delete the varLegend since it is being held by varLegendRefPtr.
    //
}


// ****************************************************************************
//  Method: avtVolumePlot::ManagesOwnTransparency
//
//  Purpose:
//      Declares whether or not this plot manages its own transparency.
//      This is one factor in the decision of whether or not to use Ice-T.
//
//  Programmer: Hank Childs
//  Creation:   February 21, 2010
//
// ****************************************************************************

bool
avtVolumePlot::ManagesOwnTransparency(void)
{
    return !PlotIsImageBased();
}


// ****************************************************************************
//  Method: avtVolumePlot::PlotIsImageBased
//
//  Purpose:
//      Determines if the plot is image based, meaning that it can't run as a
//      standard SR plot.
//
//  Programmer: Hank Childs
//  Creation:   November 24, 2004
//
//  Modifications:
//
//    Brad Whitlock, Wed Dec 15 09:33:40 PDT 2004
//    I made it use the renderer type instead.
//
//    Hank Childs, Mon Sep 11 14:50:28 PDT 2006
//    Add support for the integration ray function.
//
// ****************************************************************************

bool
avtVolumePlot::PlotIsImageBased(void)
{
    return (atts.GetRendererType() == VolumeAttributes::RayCasting ||
            atts.GetRendererType() == VolumeAttributes::RayCastingSLIVR ||
            atts.GetRendererType() == VolumeAttributes::RayCastingIntegration);
}


// ****************************************************************************
//  Method:  avtVolumePlot::Create
//
//  Purpose:
//    Call the constructor.
//
//  Programmer:  Jeremy Meredith
//  Creation:    March 27, 2001
//
// ****************************************************************************

avtPlot*
avtVolumePlot::Create()
{
    return new avtVolumePlot;
}

// ****************************************************************************
//  Method: avtVolumePlot::SetAtts
//
//  Purpose:
//    Sets the plot with the attributes.
//
//  Arguments:
//    atts : The attributes for this isoVolume plot.
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
//  Modifications:
//    Brad Whitlock, Thu Apr 19 16:06:43 PST 2001
//    Added code to set the legend.
//
//    Jeremy Meredith, Tue Jun  5 20:33:13 PDT 2001
//    Added code to set a flag if the plot needs recalculation.
//
//    Hank Childs, Fri Dec 14 09:02:05 PST 2001
//    Set the legend range correctly using artificial limits if appropriate.
//
//    Eric Brugger, Wed Jul 16 11:35:47 PDT 2003
//    Modified to work with the new way legends are managed.
//
//    Hank Childs, Tue Feb  6 15:12:04 PST 2007
//    Have the legend reflect the scaling (log, linear, skew).  Also make sure
//    that accurate numbers are put the legends range.
//
// ****************************************************************************

void
avtVolumePlot::SetAtts(const AttributeGroup *a)
{
    renderer->SetAtts(a);
    needsRecalculation =
        atts.ChangesRequireRecalculation(*(const VolumeAttributes*)a);
    atts = *(const VolumeAttributes*)a;

    SetLegendOpacities();

    double min = 0., max = 1.;
    if (*(mapper->GetInput()) != NULL)
        mapper->GetRange(min, max);
    if (atts.GetUseColorVarMin())
    {
        min = atts.GetColorVarMin();
    }
    if (atts.GetUseColorVarMax())
    {
        max = atts.GetColorVarMax();
    }
    varLegend->SetRange(min, max);
    if (atts.GetScaling() == VolumeAttributes::Linear)
        varLegend->SetScaling(0);
    else if (atts.GetScaling() == VolumeAttributes::Log)
        varLegend->SetScaling(1);
    else if (atts.GetScaling() == VolumeAttributes::Skew)
        varLegend->SetScaling(2, atts.GetSkewFactor());

    SetLegend(atts.GetLegendFlag());
}

// ****************************************************************************
// Method: avtVolumePlot::SetLegendOpacities
//
// Purpose:
//   Copies the opacity values from the attributes into the legend.
//
// Programmer: Brad Whitlock
// Creation:   Fri Apr 20 12:14:18 PDT 2001
//
// Modifications:
//
//   Hank Childs, Wed Aug 15 10:13:58 PDT 2001
//   Also set attributes for color table with the legend.
//
//   Kathleen Bonnell, Fri Aug 31 08:50:30 PDT 2001
//   Set colors in avtLUT instead of legend.  Use avtLUT to set the
//   legend's lut.
//
//   Brad Whitlock, Thu Sep 6 11:25:26 PDT 2001
//   Modified to account for changes in VolumeAttributes.
//
//   Brad Whitlock, Thu Nov 21 15:15:03 PST 2002
//   The GetColors method moved into the color control point list.
//
// ****************************************************************************

void
avtVolumePlot::SetLegendOpacities()
{
    unsigned char opacity[256];

    // Get the opacity values from the attributes.
    atts.GetOpacities(opacity);
    varLegend->SetLegendOpacities(opacity);

    unsigned char rgb[256*3];
    atts.GetColorControlPoints().GetColors(rgb, 256);
    avtLUT->SetLUTColors(rgb, 256);
    varLegend->SetLookupTable(avtLUT->GetLookupTable());
}


// ****************************************************************************
//  Method: avtVolumePlot::GetNumberOfStagesForImageBasedPlot
//
//  Purpose:
//      Determines the number of stages for this plot.
//
//  Programmer: Hank Childs
//  Creation:   December 4, 2005
//
// ****************************************************************************

int
avtVolumePlot::GetNumberOfStagesForImageBasedPlot(const WindowAttributes &a) const
{
    return volumeImageFilter->GetNumberOfStages(a);
}

// ****************************************************************************
//  Method: avtVolumePlot::ImageExecute
//
//  Purpose:
//      Takes an input image and adds a volume rendering to it.
//
//  Programmer: Hank Childs
//  Creation:   November 24, 2004
//
// ****************************************************************************

avtImage_p
avtVolumePlot::ImageExecute(avtImage_p input,
                            const WindowAttributes &window_atts)
{
    avtImage_p rv = input;

    if (volumeImageFilter != NULL)
    {
        volumeImageFilter->SetAttributes(atts);
        rv = volumeImageFilter->RenderImage(input, window_atts);
    }
    else
    {
        EXCEPTION0(ImproperUseException);
    }

    return rv;
}


// ****************************************************************************
//  Method: avtVolumePlot::SetLegend
//
//  Purpose:
//    Turns the legend on or off.
//
//  Arguments:
//    legendOn : true if the legend should be turned on, false otherwise.
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
// ****************************************************************************

void
avtVolumePlot::SetLegend(bool legendOn)
{
    if (legendOn)
    {
        varLegend->LegendOn();
    }
    else
    {
        varLegend->LegendOff();
    }
}

// ****************************************************************************
//  Method: avtVolumePlot::GetMapper
//
//  Purpose:
//      Gets the mapper as its base class (avtMapper) for our base
//      class (avtPlot).
//
//  Returns:    The mapper for this plot.
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
// ****************************************************************************

avtMapper *
avtVolumePlot::GetMapper(void)
{
    return mapper;
}

// ****************************************************************************
//  Method: avtVolumePlot::ApplyOperators
//
//  Purpose:
//      Performs the implied operators for an isoVolume plot, namely,
//      an avtVolumeFilter.
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the volume plot.
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
//  Modifications:
//
//    Jeremy Meredith, Fri Apr  6 10:56:18 PDT 2001
//    Made it use a 50k target point value so we are guaranteed square voxels.
//
//    Hank Childs, Fri Jun 15 09:16:54 PDT 2001
//    Use more general data objects as input and output.
//
//    Jeremy Meredith, Tue Nov 13 11:45:10 PST 2001
//    Added setting of target value from the plot attributes.
//
//    Hank Childs, Tue Nov 20 17:45:22 PST 2001
//    Add volume filter as an implied operator.
//
//    Hank Childs, Fri Feb  8 19:35:50 PST 2002
//    Add shift centering as an implied operator.
//
//    Hank Childs, Wed Nov 24 17:03:44 PST 2004
//    No longer apply the volume filter as an operator.
//
//    Sean Ahern, Wed Sep 10 13:25:54 EDT 2008
//    For ease of code reading and maintenance, I forced the
//    avtShiftCenteringFilter to take avtCentering type, rather than
//    int.
//
// ****************************************************************************

avtDataObject_p
avtVolumePlot::ApplyOperators(avtDataObject_p input)
{
    avtDataObject_p dob = input;

    //
    // Clean up any old filters.
    //
    if (shiftCentering != NULL)
    {
        delete shiftCentering;
        shiftCentering = NULL;
    }

    //
    // Only shift the centering if the toggle is set.
    //
    if (atts.GetSmoothData())
    {
        shiftCentering = new avtShiftCenteringFilter(AVT_NODECENT);
        shiftCentering->SetInput(dob);
        dob = shiftCentering->GetOutput();
    }

    return dob;
}


// ****************************************************************************
//  Method: GetLogicalBounds
//
//  Purpose:
//      Added for no resampling for VTK_RECTILINEAR_GRID data of more than 1 leaf
//
// ****************************************************************************

bool GetLogicalBounds(avtDataObject_p input,int &width,int &height, int &depth)
{
    const avtDataAttributes &datts = input->GetInfo().GetAttributes();
    std::string db = input->GetInfo().GetAttributes().GetFullDBName();

    debug5<<"datts->GetTime(): "<<datts.GetTime()<<endl;
    debug5<<"datts->GetTimeIndex(): "<<datts.GetTimeIndex()<<endl;
    debug5<<"datts->GetCycle(): "<<datts.GetCycle()<<endl;

    ref_ptr<avtDatabase> dbp = avtCallback::GetDatabase(db, datts.GetTimeIndex(), NULL);
    avtDatabaseMetaData *md = dbp->GetMetaData(datts.GetTimeIndex(), 1);
    std::string mesh = md->MeshForVar(datts.GetVariableName());
    const avtMeshMetaData *mmd = md->GetMesh(mesh);

    if (mmd->hasLogicalBounds == true)
    {
        width=mmd->logicalBounds[0];
        height=mmd->logicalBounds[1];
        depth=mmd->logicalBounds[2];

        return true;
    }

    return false;
}

// ****************************************************************************
//  Method: avtVolumePlot::DataMustBeResampled
//
//  Purpose:
//      Some types of data MUST be resampled or they cannot be rendered.
//
//  Notes:
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    bool
//
//  Programmer: Cameron Christensen
//  Creation:   Thursday, September 05, 2013
//
//  Modifications:
//
//    Burlen Loring, Mon Oct  5 06:42:17 PDT 2015
//    Catch the exception that can occur when using
//    an operator that produces a variable that doesn't
//    exist in the database.
//
// ****************************************************************************

bool DataMustBeResampled(avtDataObject_p input)
{
    //NOTE: We currently rely on the existence of logical bounds to
    //      determine if data must be resampled or not. There are
    //      likely other cases.

    int width,height,depth;
    try
    {
        return GetLogicalBounds(input,width,height,depth);
    }
    catch(...)
    {
        return false;
    }
}

// ****************************************************************************
//  Method: avtVolumePlot::ApplyRenderingTransformation
//
//  Purpose:
//      Performs the rendering transforamtion for a Volume plot.
//
//  Notes:
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The input data object.
//
//  Programmer: Kathleen Bonnell
//  Creation:   October 22, 2002
//
//  Modifications:
//    Hank Childs, Wed Nov 24 17:03:44 PST 2004
//    Apply the rendering transformation -- resampling in hardware, volume
//    filter for ray tracing.
//
//    Brad Whitlock, Wed Dec 15 09:34:33 PDT 2004
//    I made it use the renderer type instead.
//
//    Hank Childs, Mon Sep 11 14:50:28 PDT 2006
//    Added support for the integration ray function.
//
//    Brad Whitlock, Mon Dec 15 15:52:58 PST 2008
//    Added a filter for the HW accelerated case.
//
//    Hank Childs, Wed Dec 31 13:47:37 PST 2008
//    Renamed ResampleAtts to InternalResampleAtts.
//
//    Allen Harvey, Thurs Nov 3 7:21:13 EST 2011
//    Make resampling optional.
//
//    Brad Whitlock, Tue Jan 31 11:22:01 PST 2012
//    Add compact tree filter for non-resampling case.
//
//    Hank Childs, Tue Apr 10 17:03:40 PDT 2012
//    Create gradient as part of the volume plot (instead of cueing EEF to
//    do it).  This will allow for the volume plot to work on variables
//    that are created mid-pipeline.
//
//
//    Cyrus Harrison, Tue Mar 12 16:30:45 PDT 2013
//    Don't calc gradient for raycasting integration, lighting isn't applied
//    for this case.
//
// ****************************************************************************

avtDataObject_p
avtVolumePlot::ApplyRenderingTransformation(avtDataObject_p input)
{
    //
    // Clean up any old filters.
    //
    if (volumeFilter != NULL)
    {
        delete volumeFilter;
        volumeFilter = NULL;
    }
    if (gradientFilter != NULL)
    {
        delete gradientFilter;
        gradientFilter = NULL;
    }
    if (volumeImageFilter != NULL)
    {
        delete volumeImageFilter;
        volumeImageFilter = NULL;
    }
    if (resampleFilter != NULL)
    {
        delete resampleFilter;
        resampleFilter = NULL;
    }
    if (compactTree != NULL)
    {
        delete compactTree;
        compactTree = NULL;
    }
    avtDataObject_p dob = input;

    if (atts.GetRendererType() == VolumeAttributes::RayCasting ||
        atts.GetRendererType() == VolumeAttributes::RayCastingIntegration ||
        atts.GetRendererType() == VolumeAttributes::RayCastingSLIVR)
    {
#ifdef ENGINE
        // gradient calc for raycasting integration not needed, but
        // lighting flag may still be on
        if (atts.GetRendererType() == VolumeAttributes::RayCasting &&
            atts.GetLightingFlag())
        {
            char gradName[128], gradName2[128];
            const char *gradvar = atts.GetOpacityVariable().c_str();
            if (strcmp(gradvar, "default") == 0)
            {
                if (atts.GetScaling() == VolumeAttributes::Log ||
                    atts.GetScaling() == VolumeAttributes::Skew)
                {
                    SNPRINTF(gradName2, 128, "_expr_%s", varname);
                    gradvar = gradName2;
                }
                else
                    gradvar = varname;
            }
            // The avtVolumeFilter uses this exact name downstream.
            SNPRINTF(gradName, 128, "_%s_gradient", gradvar);

            gradientFilter = new avtGradientExpression();
            gradientFilter->SetInput(input);
            gradientFilter->SetAlgorithm(FAST);
            gradientFilter->SetOutputVariableName(gradName);
            gradientFilter->AddInputVariableName(gradvar);

            // prevent this intermediate object from getting cleared out, so
            // it is still there when we want to render.
            gradientFilter->GetOutput()->SetTransientStatus(false);
            dob = gradientFilter->GetOutput();
        }
#endif

        volumeImageFilter = new avtVolumeFilter();
        volumeImageFilter->SetAttributes(atts);
        volumeImageFilter->SetInput(dob);
        dob = volumeImageFilter->GetOutput();
    }
    else // not ray casting pipeline
    {
        //User can force resampling
        bool forceResample = atts.GetResampleFlag();

        if (DataMustBeResampled(input) || forceResample)
        {
            // Resample the data
            InternalResampleAttributes resampleAtts;
            resampleAtts.SetDistributedResample(false);
            resampleAtts.SetTargetVal(atts.GetResampleTarget());
            resampleAtts.SetPrefersPowersOfTwo(atts.GetRendererType() == VolumeAttributes::Texture3D);
            resampleAtts.SetUseTargetVal(true);

            // Unless user forced resampling use actual logical bounds if they exist.
            if (!forceResample)
            {
                int width,height,depth;
                if (GetLogicalBounds(input,width,height,depth))
                {
                    resampleAtts.SetWidth(width);
                    resampleAtts.SetHeight(height);
                    resampleAtts.SetDepth(depth);
                    resampleAtts.SetUseTargetVal(false);
                }
            }

            resampleFilter = new avtResampleFilter(&resampleAtts);
            resampleFilter->SetInput(input);

            dob = resampleFilter->GetOutput();
        }
        else
        {
            // Combine multiple domains into a single domain.
            compactTree = new avtCompactTreeFilter();
            compactTree->SetInput(input);
            compactTree->SetParallelMerge(true);
            compactTree->SetCompactDomainsMode(avtCompactTreeFilter::Always);
            dob = compactTree->GetOutput();
        }

        // Apply a filter that will work on the combined data to make histograms.
        volumeFilter = new avtLowerResolutionVolumeFilter();
        volumeFilter->SetAtts(&atts);
        volumeFilter->SetInput(dob);
        dob = volumeFilter->GetOutput();
    }
    return dob;
}


// ****************************************************************************
//  Method: avtVolumePlot::CustomizeBehavior
//
//  Purpose:
//      Customizes the behavior of the output.
//
//  Programmer: Jeremy Meredith
//  Creation:   March 27, 2001
//
//  Modifications:
//    Brad Whitlock, Thu Apr 19 16:15:23 PST 2001
//    Added a legend.
//
//    Kathleen Bonnell, Thu Sep 18 13:40:26 PDT 2003
//    Set anti-aliased render order to be the same as normal.
//
//    Hank Childs, Tue Nov 30 09:03:00 PST 2004
//    Make sure legend always reflects artificial ranges.
//
// ****************************************************************************

void
avtVolumePlot::CustomizeBehavior(void)
{
    // make it go last
    behavior->SetRenderOrder(MUST_GO_LAST);
    behavior->SetAntialiasedRenderOrder(MUST_GO_LAST);

    // Add a legend.
    double min, max;
    mapper->GetRange(min, max);
    varLegend->SetVarRange(min, max);
    if (atts.GetUseColorVarMin())
    {
        min = atts.GetColorVarMin();
    }
    if (atts.GetUseColorVarMax())
    {
        max = atts.GetColorVarMax();
    }
    varLegend->SetRange(min, max);
    behavior->SetLegend(varLegendRefPtr);
}


// ****************************************************************************
//  Method: avtVolumePlot::EnhanceSpecification
//
//  Purpose:
//      This is a hook from the base class that allows the plot to add to the
//      specification.  It is needed for this derived type because we want
//      to (potentially) specify that another variable gets read in for the
//      opacity.
//
//  Programmer: Hank Childs
//  Creation:   November 15, 2001
//
//  Modifications:
//    Hank Childs, Thu Aug 26 17:44:13 PDT 2010
//    Calculate the extents for the opacity variable.
//
//    Brad Whitlock, Mon Jan 30 14:00:46 PST 2012
//    Add support for "compact" variable, which seems to be a point radius
//    for splatting volume renderer.
//
// ****************************************************************************

avtContract_p
avtVolumePlot::EnhanceSpecification(avtContract_p spec)
{
    std::string ov = atts.GetOpacityVariable();
    std::string cv = atts.GetCompactVariable();
    if (ov == "default")
    {
        if(atts.GetResampleFlag())
            return spec;
        else if(cv == "default")
        {
            // We're not resampling so we can return the original specification
            // if our compact variable is set to "default".
            return spec;
        }
    }
    avtDataRequest_p ds = spec->GetDataRequest();
    std::string primaryVariable(ds->GetVariable());
    if (ov == primaryVariable)
    {
         if(atts.GetResampleFlag())
         {
             //
             // They didn't leave it as "default", but it is the same variable, so
             // don't read it in again.
             //
             return spec;
         }
         else if(cv == primaryVariable)
         {
             // We're not resampling and the compact variable was the same as
             // the primary variable, so don't read it again.
             return spec;
         }
    }

    //
    // The pipeline specification should really be const -- it is used
    // elsewhere, so we can't modify it and return it.  Make a copy and in
    // the new copy, add a secondary variable.
    //
    avtDataRequest_p nds = new avtDataRequest(primaryVariable.c_str(),
                                              ds->GetTimestep(), ds->GetRestriction());
    if(ov != "default")
    {
        debug5 << "Adding secondary variable: " << ov << endl;
        nds->AddSecondaryVariable(ov.c_str());
    }
    if(cv != "default" && !atts.GetResampleFlag())
    {
        debug5 << "Adding secondary variable: " << cv << endl;
        nds->AddSecondaryVariable(cv.c_str());
    }
    avtContract_p rv = new avtContract(spec, nds);
    rv->SetCalculateVariableExtents(ov, true);

    return rv;
}


// ****************************************************************************
//  Method: avtVolumePlot::ReleaseData
//
//  Purpose:
//      Release the problem sized data associated with this plot.
//
//  Programmer: Hank Childs
//  Creation:   September 12, 2002
//
// ****************************************************************************

void
avtVolumePlot::ReleaseData(void)
{
    avtVolumeDataPlot::ReleaseData();

    if (volumeImageFilter != NULL)
    {
        volumeImageFilter->ReleaseData();
    }
    if (shiftCentering != NULL)
    {
        shiftCentering->ReleaseData();
    }
}


// ****************************************************************************
//  Method:  avtVolumePlot::Equivalent
//
//  Purpose:
//    Returns true if the plots are equivalent.
//
//  Arguments:
//    a          the atts to compare
//
//  Programmer:  Hank Childs
//  Creation:    November 20, 2001
//
//  Modifications:
//    Brad Whitlock, Tue Jan 31 17:18:59 PST 2012
//    Added compact variable.
//
// ****************************************************************************

bool
avtVolumePlot::Equivalent(const AttributeGroup *a)
{
    const VolumeAttributes *objAtts = (const VolumeAttributes *)a;
    // Almost the inverse of changes require recalculation -- doSoftware being
    // different is okay!
    if (atts.GetResampleFlag() != objAtts->GetResampleFlag())
        return false;
    if (atts.GetOpacityVariable() != objAtts->GetOpacityVariable())
        return false;
    if (atts.GetCompactVariable() != objAtts->GetCompactVariable())
        return false;
    if (atts.GetResampleTarget() != objAtts->GetResampleTarget())
        return false;
    return true;
}


