/*****************************************************************************
*
* Copyright (c) 2000 - 2017, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                             avtTensorPlot.C                               //
// ************************************************************************* //

#include <avtTensorPlot.h>

#include <vtkSphereSource.h>

#include <avtGhostZoneFilter.h>
#include <avtTensorGlyphMapper.h>
#include <avtLookupTable.h>
#include <avtTensorFilter.h>
#include <avtVariableLegend.h>


// ****************************************************************************
//  Method: avtTensorPlot constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
//  Modifications:
//    Kathleen Biagas, Thus Feb 7 11:19:41 PST 2013
//    Set sphere source's output port as input to the glyph mapper, preserves
//    pipeline connectivity with vtk-6.
//
// ****************************************************************************

avtTensorPlot::avtTensorPlot()
{
    colorsInitialized = false;
    TensorFilter = new avtTensorFilter(true, 10);
    ghostFilter  = new avtGhostZoneFilter();
    avtLUT       = new avtLookupTable();

    //
    // The tensor glyph mapper does funny things with normals.  Its best
    // just to remove the normals from the sphere source altogether.
    //
    vtkSphereSource *src = vtkSphereSource::New();
    tensorMapper  = new avtTensorGlyphMapper(src->GetOutputPort());
    // bump up the reference count
    src->Register(NULL);
    src->Delete();

    varLegend = new avtVariableLegend;
    varLegend->SetTitle("Tensor");

    //
    // This is to allow the legend to be reference counted so the behavior can
    // still access it after the plot is deleted.  The legend cannot be
    // reference counted all of the time since we need to know that it is a
    // VariableLegend.
    //
    varLegendRefPtr = varLegend;
}


// ****************************************************************************
//  Method: avtTensorPlot destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
// ****************************************************************************

avtTensorPlot::~avtTensorPlot()
{
    if (tensorMapper != NULL)
    {
        delete tensorMapper;
        tensorMapper = NULL;
    }
    if (TensorFilter != NULL)
    {
        delete TensorFilter;
        TensorFilter = NULL;
    }
    if (ghostFilter != NULL)
    {
        delete ghostFilter;
        ghostFilter = NULL;
    }
    if (avtLUT != NULL)
    {
        delete avtLUT;
        avtLUT = NULL;
    }
}


// ****************************************************************************
//  Method:  avtTensorPlot::Create
//
//  Purpose:
//    Call the constructor.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
// ****************************************************************************

avtPlot*
avtTensorPlot::Create()
{
    return new avtTensorPlot;
}


// ****************************************************************************
//  Method: avtTensorPlot::GetMapper
//
//  Purpose:
//      Gets a mapper for this plot, it is actually a variable mapper.
//
//  Returns:    The variable mapper typed as its base class mapper.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
// ****************************************************************************

avtMapper *
avtTensorPlot::GetMapper(void)
{
    return tensorMapper;
}


// ****************************************************************************
//  Method: avtTensorPlot::ApplyOperators
//
//  Purpose:
//      Applies the operators associated with a Tensor plot.  
//      The output from this method is a query-able object.
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the Tensor plot has been applied.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
//  Modifications:
//
//    Hank Childs, Tue Feb 22 13:50:57 PST 2011
//    Filters used in this routine are actually part of the rendering
//    transformation.  Move them to the ApplyRenderingTransformation method.
//
// ****************************************************************************

avtDataObject_p
avtTensorPlot::ApplyOperators(avtDataObject_p input)
{
    return input;
}


// ****************************************************************************
//  Method: avtTensorPlot::ApplyRenderingTransformation
//
//  Purpose:
//      Applies the rendering transformation associated with a Tensor plot.  
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the Tensor plot has been applied.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
//  Modifications:
//
//    Hank Childs, Tue Feb 22 13:50:57 PST 2011
//    Filters being used in ApplyOperators were doing rendering transformations.
//    Put them here.
//
// ****************************************************************************

avtDataObject_p
avtTensorPlot::ApplyRenderingTransformation(avtDataObject_p input)
{
    ghostFilter->SetInput(input);
    TensorFilter->SetInput(ghostFilter->GetOutput());
    return TensorFilter->GetOutput();
}


// ****************************************************************************
//  Method: avtTensorPlot::CustomizeBehavior
//
//  Purpose:
//      Customizes the behavior as appropriate for a Tensor plot.  This includes
//      behavior like shifting towards or away from the screen.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
// ****************************************************************************

void
avtTensorPlot::CustomizeBehavior(void)
{
    behavior->SetLegend(varLegendRefPtr);
}


// ****************************************************************************
//  Method: avtTensorPlot::CustomizeMapper
//
//  Purpose:
//      A hook from the base class that allows the plot to change its mapper
//      based on the dataset input. 
//
//  Arguments:
//      doi     The data object information.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
//  Modifications:
//    Kathleen Bonnell, Mon Sep 29 12:31:18 PDT 2003
//    Set antialiased render order.
// ****************************************************************************

void
avtTensorPlot::CustomizeMapper(avtDataObjectInformation &doi)
{
    behavior->SetRenderOrder(DOES_NOT_MATTER);
    behavior->SetAntialiasedRenderOrder(DOES_NOT_MATTER);

    //
    // Set the legend ranges.
    //
    SetLegendRanges();
}


// ****************************************************************************
//  Method: avtTensorPlot::SetAtts
//
//  Purpose:
//      Sets the atts for the Tensor plot.
//
//  Arguments:
//      atts    The attributes for this Tensor plot.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Sep 23 20:57:03 PST 2003
//
//  Modifications:
//    Eric Brugger, Wed Nov 24 13:17:56 PST 2004
//    I added scaleByMagnitude and autoScale.
//
//    Kathleen Bonnell, Mon Jan 17 18:20:03 MST 2011
//    Consider invertColorTable flag when setting updateColors.
//
// ****************************************************************************

void
avtTensorPlot::SetAtts(const AttributeGroup *a)
{
    const TensorAttributes *newAtts = (const TensorAttributes *)a;

    // See if the colors will need to be updated.
    bool updateColors = (!colorsInitialized) ||
       (atts.GetColorTableName() != newAtts->GetColorTableName()) ||
       (atts.GetInvertColorTable() != newAtts->GetInvertColorTable());

    // See if any attributes that require the plot to be regenerated were
    // changed and copy the state object.
    needsRecalculation = atts.ChangesRequireRecalculation(*newAtts);
    atts = *newAtts;

    // Set internal plot state using the information in the atts state object.
    bool useStride = atts.GetUseStride();
    if (useStride)
    {
        TensorFilter->SetStride(atts.GetStride());
    }
    else
    {
        TensorFilter->SetNTensors(atts.GetNTensors());
    }

    tensorMapper->SetScaleByMagnitude(atts.GetScaleByMagnitude());
    tensorMapper->SetAutoScale(atts.GetAutoScale());
    tensorMapper->SetScale(atts.GetScale());

    if (atts.GetColorByEigenvalues())
    {
        tensorMapper->ColorByMagOn();
    }
    else
    {
        const unsigned char *col = atts.GetTensorColor().GetColor();
        tensorMapper->ColorByMagOff(col);
    }

    // Update the plot's colors if needed.
    if (atts.GetColorByEigenvalues() &&
       (updateColors || atts.GetColorTableName() == "Default"))
    {
        colorsInitialized = true;
        SetColorTable(atts.GetColorTableName().c_str());
    }

    //
    // Update the legend.
    //
    SetLegend(atts.GetUseLegend());
}


// ****************************************************************************
//  Method: avtTensorPlot::SetColorTable
//
//  Purpose:
//      Sets the plot's color table if the color table is the same as that of
//      the plot or we are using the default color table for the plot.
//
//  Arguments:
//      ctName : The name of the color table to use.
//
//  Returns:    Returns true if the color table is updated.
//
//  Programmer: Hank Childs
//  Creation:   September 23, 2003
//
//  Modifications:
//    Kathleen Bonnell, Mon Jan 17 18:20:03 MST 2011
//    Retrieve invertColorTable flag and pass to avtLUT.
//
// ****************************************************************************

bool
avtTensorPlot::SetColorTable(const char *ctName)
{
    bool retval = false;
    if (atts.GetColorByEigenvalues())
    {
        bool namesMatch = (atts.GetColorTableName() == std::string(ctName));
        bool invert = atts.GetInvertColorTable();
        if (atts.GetColorTableName() == "Default")
        {
            retval =  avtLUT->SetColorTable(NULL, namesMatch, false, invert);
        }
        else
        {
            retval =  avtLUT->SetColorTable(ctName, namesMatch, false, invert);
        }

        if (retval)
        {
            tensorMapper->SetLookupTable(avtLUT->GetLookupTable());
        }
    }
    else
    {
        avtLUT->SetLUTColors(atts.GetTensorColor().GetColor(), 1);
    }

    return retval;
}


// ****************************************************************************
//  Method: avtTensorPlot::SetLegend
//
//  Purpose:
//    Turns the legend on or off.
//
//  Arguments:
//    legendOn  : true if the legend should be turned on, false otherwise.
//
//  Programmer: Hank Childs
//  Creation:   September 23, 2003
//
// ****************************************************************************

void
avtTensorPlot::SetLegend(bool legendOn)
{
    if (legendOn)
    {
        // Set scaling.
        varLegend->LegendOn();
        varLegend->SetLookupTable(avtLUT->GetLookupTable());
        varLegend->SetScaling();
    }
    else
    {
        varLegend->LegendOff();
    }
}


// ****************************************************************************
//  Method: avtTensorPlot::SetLegendRanges
//
//  Purpose:
//      Sets the range to use for the legend.
//
//  Programmer: Hank Childs
//  Date:       September 23, 2003
//
// ****************************************************************************

void
avtTensorPlot::SetLegendRanges()
{
    double min = 0., max = 1.;
    tensorMapper->GetRange(min, max);

    //
    // Set the range for the legend's text and colors.
    //
    varLegend->SetVarRange(min, max);
    varLegend->SetRange(min, max);
}


// ****************************************************************************
//  Method: avtTensorPlot::ReleaseData
//
//  Purpose:
//      Release the problem sized data associated with this plot.
//
//  Programmer: Hank Childs
//  Creation:   September 12, 2002
//
// ****************************************************************************

void
avtTensorPlot::ReleaseData(void)
{
    avtPointDataPlot::ReleaseData();

    if (TensorFilter != NULL)
    {
        TensorFilter->ReleaseData();
    }
    if (ghostFilter != NULL)
    {
        ghostFilter->ReleaseData();
    }
}


// ****************************************************************************
//  Method: avtTensorPlot::GetExtraInfoForPick
//
//  Purpose:
//    Override default settings for extraPickinfo.
//
//  Programmer: Kathleen Biagas
//  Creation:   February 29, 2012
//
// ****************************************************************************

const MapNode &
avtTensorPlot::GetExtraInfoForPick()
{
    extraPickInfo["glyphPickAlways"] = true;
    extraPickInfo["nodeCenteredNeedZonesForPick"] = false;

    return extraPickInfo;
}


