/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPVLookupTable.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkPVLookupTable - a combination of vtkColorTransferFunction and
// vtkLookupTable.
// .SECTION Description
// This is a cross between a vtkColorTransferFunction and a vtkLookupTable
// selectively combiniting the functionality of both.

#ifndef __vtkPVLookupTable_h
#define __vtkPVLookupTable_h

#include "vtkScalarsToColors.h"

class vtkLookupTable;
class vtkColorTransferFunction;

class VTK_EXPORT vtkPVLookupTable : public vtkScalarsToColors
{
public:
  static vtkPVLookupTable* New();
  vtkTypeRevisionMacro(vtkPVLookupTable, vtkScalarsToColors);
  void PrintSelf(ostream& os, vtkIndent indent);
  
  // Description:
  // Generate discretized lookup table, if applicable.
  virtual void Build();

  // Description:
  // Sets/Gets the range of scalars which will be mapped.
  virtual double *GetRange() { return this->TableRange; }
  virtual void SetRange(double min, double max);

  // Description:
  // Set if the values are to mapped after discretization. The
  // number of discrete values is set by using SetNumberOfValues().
  // Not set by default, i.e. color value is determined by
  // interpolating at the scalar value.
  vtkSetMacro(Discretize, int);
  vtkGetMacro(Discretize, int);
  vtkBooleanMacro(Discretize, int);

  // Description:
  // Set the number of values i.e. colors to be generated in the
  // discrete lookup table. This has no effect if Discretize is off.
  void SetNumberOfValues(vtkIdType number);
  vtkGetMacro(NumberOfValues, vtkIdType);

  // Description:
  // Map one value through the lookup table and return a color defined
  // as a RGBA unsigned char tuple (4 bytes).
  virtual unsigned char *MapValue(double v);

  // Description:
  // Map one value through the lookup table and return the color as
  // an RGB array of doubles between 0 and 1.
  virtual void GetColor(double v, double rgb[3]);

  // Description:
  // Add/Remove a point to/from the function defined in RGB or HSV
  // Return the index of the point (0 based), or -1 on error.
  // Unlike vtkColorTransferFunction, these methods expect that
  // the x is normalized [0,1].
  int AddRGBPoint( double x, double r, double g, double b );
  int AddRGBPoint( double x, double r, double g, double b, 
                   double midpoint, double sharpness );
  int AddHSVPoint( double x, double h, double s, double v );
  int AddHSVPoint( double x, double h, double s, double v,
                   double midpoint, double sharpness );

  // Description:
  // Remove all points
  void RemoveAllPoints();

  // Description:
  // Set/Get the color space used for interpolation: RGB, or HSV.
  // In HSV mode, if HSVWrap is on, it  will take the shortest path in Hue
  // (going back through 0 if that is the shortest way around the hue circle)
  // whereas if HSVWrap is off it will not go through 0 (in order the match
  // the current functionality of vtkLookupTable)
  void SetColorSpace(int);
  void SetHSVWrap(int);

  // Description:
  // An internal method typically not used in applications.
  virtual void MapScalarsThroughTable2(void *input, unsigned char *output,
                                       int inputDataType, int numberOfValues,
                                       int inputIncrement, 
                                       int outputFormat);
protected:
  vtkPVLookupTable();
  ~vtkPVLookupTable();

  vtkSetVector2Macro(TableRange, double);
  int Discretize;
  double TableRange[2];
  vtkIdType NumberOfValues;
  vtkLookupTable* LookupTable;
  vtkColorTransferFunction* ColorTransferFunction;

  vtkTimeStamp BuildTime;
private:
  vtkPVLookupTable(const vtkPVLookupTable&); // Not implemented.
  void operator=(const vtkPVLookupTable&); // Not implemented.
};

#endif

