/*=========================================================================

  Program:   ParaView
  Module:    vtkPVColorMapUI.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVColorMapUI.h"

#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWEntry.h"
#include "vtkKWFrameWithLabel.h"
#include "vtkKWLabel.h"
#include "vtkKWMenu.h"
#include "vtkKWMenuButton.h"
#include "vtkKWMenuButtonWithLabel.h"
#include "vtkKWPushButton.h"
#include "vtkKWRange.h"
#include "vtkKWScaleWithEntry.h"
#include "vtkKWThumbWheel.h"
#include "vtkKWTkUtilities.h"
#include "vtkLookupTable.h"
#include "vtkObjectFactory.h"
#include "vtkPVApplication.h"
#include "vtkPVArrayInformation.h"
#include "vtkPVColorMap.h"
#include "vtkPVDataSetAttributesInformation.h"
#include "vtkPVGeometryInformation.h"
#include "vtkPVSource.h"
#include "vtkPVSourceCollection.h"
#include "vtkPVTextPropertyEditor.h"
#include "vtkPVTraceHelper.h"
#include "vtkPVWindow.h"
#include "vtkSMDataObjectDisplayProxy.h"
#include "vtkTextProperty.h"

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPVColorMapUI);
vtkCxxRevisionMacro(vtkPVColorMapUI, "1.9");

vtkCxxSetObjectMacro(vtkPVColorMapUI, CurrentColorMap, vtkPVColorMap);

/* 
 * This part was generated by ImageConvert from image:
 *    presets.png (zlib, base64)
 */
#define image_presets_width         16
#define image_presets_height        16
#define image_presets_pixel_size    4
#define image_presets_length        264

static unsigned char image_presets[] = 
  "eNrdkusNhSAMhZkHhtAlXAKHgCFkCZeAIXAIXeJcikiqQaN/L8kJr349LQGAQJG1FleB3T"
  "+JYkMIecvnNznuWJ7jqbay/jSIEULgzIvHOvjc4t+yR0//4P9VV3+SUgp932MYBoxaw6S7"
  "yTnM84zgA5ZlwbZtObblr6RMfJd5PSbeGDg3Zd4TH2Pl+f858d3ur/XB7/4+eETGt6SUrP"
  "WTP+WuvPfJn+pf63tz0Zm89G9N4qfSf/Kn/tfi3+K/vf+Z/wENH5fO";

#define MAP_HEIGHT_INITIAL 25
#define MAP_WIDTH_INITIAL 20

//----------------------------------------------------------------------------
vtkPVColorMapUI::vtkPVColorMapUI()
{
  this->CurrentColorMap = NULL;

  this->ScalarColorBarFrame = vtkKWFrameWithLabel::New();
  this->VisibilityCheck = vtkKWCheckButton::New();
  this->DefaultCheck = vtkKWCheckButton::New();
  this->ParameterFrame = vtkKWFrame::New();
  this->ParameterLabel = vtkKWLabel::New();
  this->ParameterMenu = vtkKWMenuButton::New();
  this->VectorLabel = vtkKWLabel::New();
  this->VectorModeMenu = vtkKWMenuButton::New();
  this->VectorComponentMenu = vtkKWMenuButton::New();
  this->ScalarRangeFrame = vtkKWFrame::New();
  this->ScalarRangeLockCheck = vtkKWCheckButton::New();
  this->ScalarRangeWidget = vtkKWRange::New();
  this->ScalarRangeWidget->ClampRangeOff();
  this->NumberOfColorsScale = vtkKWScaleWithEntry::New();
  this->ColorEditorFrame = vtkKWFrame::New();
  this->PresetsMenuButton = vtkKWMenuButton::New();
  this->StartColorButton = vtkKWChangeColorButton::New();
  this->Map = vtkKWLabel::New();
  this->EndColorButton = vtkKWChangeColorButton::New();
  this->OutOfRangeFrame = vtkKWFrame::New();
  this->OutOfRangeLabel = vtkKWLabel::New();
  this->OutOfRangeCheck = vtkKWCheckButton::New();
  this->LowColorLabel = vtkKWLabel::New();
  this->LowColorButton = vtkKWChangeColorButton::New();
  this->HighColorLabel = vtkKWLabel::New();
  this->HighColorButton = vtkKWChangeColorButton::New();
  this->ScalarBarTitleFrame = vtkKWFrame::New();
  this->ScalarBarTitleLabel = vtkKWLabel::New();
  this->ScalarBarTitleEntry = vtkKWEntry::New();
  this->ScalarBarVectorTitleEntry = vtkKWEntry::New();
  this->TitleTextPropertyWidget = vtkPVTextPropertyEditor::New();
  this->ScalarBarLabelFormatFrame = vtkKWFrame::New();
  this->ScalarBarLabelFormatLabel = vtkKWLabel::New();
  this->ScalarBarLabelFormatEntry = vtkKWEntry::New();
  this->LabelTextPropertyWidget = vtkPVTextPropertyEditor::New();
  this->NumberOfLabelsThumbWheel = vtkKWThumbWheel::New();
  this->BackButton = vtkKWPushButton::New();
}

//----------------------------------------------------------------------------
vtkPVColorMapUI::~vtkPVColorMapUI()
{
  this->SetCurrentColorMap(NULL);

  this->ScalarColorBarFrame->Delete();
  this->VisibilityCheck->Delete();
  this->DefaultCheck->Delete();
  this->ParameterFrame->Delete();
  this->ParameterLabel->Delete();
  this->ParameterMenu->Delete();
  this->VectorLabel->Delete();
  this->VectorModeMenu->Delete();
  this->VectorComponentMenu->Delete();
  this->ScalarRangeFrame->Delete();
  this->ScalarRangeLockCheck->Delete();
  this->ScalarRangeWidget->Delete();
  this->NumberOfColorsScale->Delete();
  this->ColorEditorFrame->Delete();
  this->PresetsMenuButton->Delete();
  this->StartColorButton->Delete();
  this->Map->Delete();
  this->EndColorButton->Delete();
  this->OutOfRangeFrame->Delete();
  this->OutOfRangeLabel->Delete();
  this->OutOfRangeCheck->Delete();
  this->LowColorLabel->Delete();
  this->LowColorButton->Delete();
  this->HighColorLabel->Delete();
  this->HighColorButton->Delete();
  this->ScalarBarTitleFrame->Delete();
  this->ScalarBarTitleLabel->Delete();
  this->ScalarBarTitleEntry->Delete();
  this->ScalarBarVectorTitleEntry->Delete();
  this->TitleTextPropertyWidget->Delete();
  this->ScalarBarLabelFormatFrame->Delete();
  this->ScalarBarLabelFormatLabel->Delete();
  this->ScalarBarLabelFormatEntry->Delete();
  this->LabelTextPropertyWidget->Delete();
  this->NumberOfLabelsThumbWheel->Delete();
  this->BackButton->Delete();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::CreateWidget()
{
  if (this->IsCreated())
    {
    vtkErrorMacro(<< this->GetClassName() << " already created");
    return;
    }

  this->Superclass::CreateWidget();

  const char *grid_settings = "-padx 1 -pady 2";
  int index;

  this->ScalarColorBarFrame->SetParent(this);
  this->ScalarColorBarFrame->Create();
  this->ScalarColorBarFrame->SetLabelText("Scalar Color Bar");

  // Scalar bar : Visibility
  this->VisibilityCheck->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->VisibilityCheck->Create();
  this->VisibilityCheck->SetText("Visibility");
  this->VisibilityCheck->SetCommand(this, "VisibilityCheckCallback");
  this->VisibilityCheck->SetBalloonHelpString(
    "Toggle the visibility of the scalar color bar.");
  this->VisibilityCheck->EnabledOff();

  // Store default values
  this->DefaultCheck->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->DefaultCheck->Create();
  this->DefaultCheck->SetText("Use as default");
  this->DefaultCheck->SetCommand(this, "DefaultCheckCallback");
  this->DefaultCheck->SetBalloonHelpString(
    "Indicate whether to use this set of parameters as the default.");

  // Parameter frame
  this->ParameterFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->ParameterFrame->Create();

  this->ParameterLabel->SetParent(this->ParameterFrame);
  this->ParameterLabel->Create();
  this->ParameterLabel->SetText("Parameter");

  this->ParameterMenu->SetParent(this->ParameterFrame);
  this->ParameterMenu->Create();
//  this->ParameterMenu->IndicatorVisibilityOn();
  this->ParameterMenu->SetBalloonHelpString(
    "Select the parameter for which to display a scalar color bar.");

  // Vector menus
  this->VectorLabel->SetParent(this->ParameterFrame);
  this->VectorLabel->SetText("Vector");
  this->VectorLabel->Create();
  this->VectorLabel->EnabledOff();
  
  this->VectorModeMenu->SetParent(this->ParameterFrame);
  this->VectorModeMenu->Create();
  this->VectorModeMenu->GetMenu()->AddRadioButton(
    "Magnitude", this, "VectorModeMagnitudeCallback");
  this->VectorModeMenu->GetMenu()->AddRadioButton(
    "Component", this, "VectorModeComponentCallback");
  this->VectorModeMenu->SetValue("Magnitude");
  this->VectorModeMenu->EnabledOff();

  this->VectorComponentMenu->SetParent(this->ParameterFrame);
  this->VectorComponentMenu->Create();
  this->VectorComponentMenu->EnabledOff();

  this->Script("grid %s -row 0 -column 0 -sticky w %s",
               this->ParameterLabel->GetWidgetName(), grid_settings);
  this->Script("grid %s -row 0 -column 1 -columnspan 2 -sticky ew %s",
               this->ParameterMenu->GetWidgetName(), grid_settings);
  this->Script("grid columnconfigure %s 1 -weight 2",
               this->ParameterFrame->GetWidgetName());
  this->Script("grid %s -row 1 -column 0 -sticky w %s",
               this->VectorLabel->GetWidgetName(), grid_settings);
  this->Script("grid %s -row 1 -column 1 -sticky ew %s",
               this->VectorModeMenu->GetWidgetName(), grid_settings);
  this->Script("grid %s -row 1 -column 2 -sticky ew %s",
               this->VectorComponentMenu->GetWidgetName(), grid_settings);

  // Scalar range frame
  this->ScalarRangeFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->ScalarRangeFrame->Create();

  this->ScalarRangeLockCheck->SetParent(this->ScalarRangeFrame);
  this->ScalarRangeLockCheck->Create();
  this->ScalarRangeLockCheck->SetConfigurationOption(
    "-image", "PVUnlockedButton");
  this->ScalarRangeLockCheck->SetConfigurationOption(
    "-selectimage", "PVLockedButton");
  this->ScalarRangeLockCheck->SetReliefToFlat();
  this->ScalarRangeLockCheck->SetSelectedState(0);
  this->ScalarRangeLockCheck->IndicatorVisibilityOff();
  this->ScalarRangeLockCheck->SetBalloonHelpString(
    "Lock the range so that ParaView will not change it.");
  this->ScalarRangeLockCheck->SetCommand(this, "LockCheckCallback");
  this->Script("pack %s -side left -expand f",
               this->ScalarRangeLockCheck->GetWidgetName());

  this->ScalarRangeWidget->SetParent(this->ScalarRangeFrame);
  this->ScalarRangeWidget->Create();
  this->ScalarRangeWidget->SetWholeRange(
    -VTK_LARGE_FLOAT, VTK_LARGE_FLOAT);
  this->ScalarRangeWidget->EntriesVisibilityOn();
  this->ScalarRangeWidget->LabelVisibilityOff();
  this->ScalarRangeWidget->GetEntry1()->SetWidth(7);
  this->ScalarRangeWidget->GetEntry2()->SetWidth(7);
  this->ScalarRangeWidget->SetCommand(this, "ScalarRangeWidgetCallback");
  this->ScalarRangeWidget->SetEntry1PositionToLeft();
  this->ScalarRangeWidget->SetEntry2PositionToRight();
  this->ScalarRangeWidget->SetBalloonHelpString(
    "Set the minimum and maximum of the values of the color map");
  this->Script("pack %s -side left -fill x -expand t",
               this->ScalarRangeWidget->GetWidgetName());

  // Color editor frame
  this->ColorEditorFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->ColorEditorFrame->Create();

  this->StartColorButton->SetParent(this->ColorEditorFrame);
  this->StartColorButton->LabelVisibilityOff();
  this->StartColorButton->Create();
  this->StartColorButton->SetColor(1.0, 0.0, 0.0);
  this->StartColorButton->SetCommand(this, "StartColorButtonCallback");
  this->StartColorButton->SetBalloonHelpString("Select the minimum color.");

  this->Map->SetParent(this->ColorEditorFrame);
  this->Map->Create();
  this->Map->SetBorderWidth(0);
  this->Map->SetPadX(0);
  this->Map->SetPadY(0);

  this->Script("bind %s <Configure> {%s MapConfigureCallback %s}", 
               this->Map->GetWidgetName(), 
               this->GetTclName(), "%w %h");

  this->EndColorButton->SetParent(this->ColorEditorFrame);
  this->EndColorButton->LabelVisibilityOff();
  this->EndColorButton->Create();
  this->EndColorButton->SetColor(0.0, 0.0, 1.0);
  this->EndColorButton->SetCommand(this, "EndColorButtonCallback");
  this->EndColorButton->SetBalloonHelpString("Select the maximum color.");

  this->PresetsMenuButton->SetParent(this->ColorEditorFrame);
  this->PresetsMenuButton->Create();
  this->PresetsMenuButton->IndicatorVisibilityOff();
  this->PresetsMenuButton->SetBalloonHelpString("Select a preset color map.");

  index = this->PresetsMenuButton->GetMenu()->AddCommand(
    "Blue to Red", this, "SetColorSchemeToBlueRed");
  this->PresetsMenuButton->GetMenu()->SetItemHelpString(
    index, "Set Color Scheme to Blue-Red");

  index = this->PresetsMenuButton->GetMenu()->AddCommand(
    "Red to Blue", this, "SetColorSchemeToRedBlue");
  this->PresetsMenuButton->GetMenu()->SetItemHelpString(
    index, "Set Color Scheme to Red-Blue");

  index = this->PresetsMenuButton->GetMenu()->AddCommand(
    "Grayscale", this, "SetColorSchemeToGrayscale");
  this->PresetsMenuButton->GetMenu()->SetItemHelpString(
    index, "Set Color Scheme to Grayscale");

  index = this->PresetsMenuButton->GetMenu()->AddCommand(
    "CIELab Blue to Red", this, "SetColorSchemeToLabBlueRed");
  this->PresetsMenuButton->GetMenu()->SetItemHelpString(
    index, "Set Color Scheme to Lab Blue To Red");
  
  this->PresetsMenuButton->SetImageToPixels(image_presets, 
                                            image_presets_width, 
                                            image_presets_height, 
                                            image_presets_pixel_size,
                                            image_presets_length);

  this->Script("grid %s %s %s %s -sticky news -padx 1 -pady 2",
               this->PresetsMenuButton->GetWidgetName(),
               this->StartColorButton->GetWidgetName(),
               this->Map->GetWidgetName(),
               this->EndColorButton->GetWidgetName());

  this->Script("grid columnconfigure %s 2 -weight 1",
               this->Map->GetParent()->GetWidgetName());
  this->Script("grid %s -ipadx 1",
               this->PresetsMenuButton->GetWidgetName());

  // Color map resolution
  this->NumberOfColorsScale->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->NumberOfColorsScale->Create();
  this->NumberOfColorsScale->SetRange(2, 256);
  this->NumberOfColorsScale->SetLabelText("Resolution:");
  this->NumberOfColorsScale->SetEndCommand(this, 
                                           "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetEntryCommand(this, 
                                             "NumberOfColorsScaleCallback");
  this->NumberOfColorsScale->SetBalloonHelpString(
    "Select the discrete number of colors in the color map.");

  // Out of range frame
  this->OutOfRangeFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->OutOfRangeFrame->Create();

  this->OutOfRangeLabel->SetParent(this->OutOfRangeFrame);
  this->OutOfRangeLabel->SetText("Out of Range:");
  this->OutOfRangeLabel->Create();

  this->OutOfRangeCheck->SetParent(this->OutOfRangeFrame);
  this->OutOfRangeCheck->Create();
  this->OutOfRangeCheck->SetCommand(this, "OutOfRangeCheckCallback");

  this->LowColorLabel->SetParent(this->OutOfRangeFrame);
  this->LowColorLabel->SetText("Low");
  this->LowColorLabel->Create();
  this->LowColorLabel->EnabledOff();

  this->LowColorButton->SetParent(this->OutOfRangeFrame);
  this->LowColorButton->Create();
  this->LowColorButton->LabelVisibilityOff();
  this->LowColorButton->SetColor(0, 0, 0);
  this->LowColorButton->SetCommand(this, "LowColorButtonCallback");
  this->LowColorButton->SetBalloonHelpString("Select the color for values below the low end of the scalar range.");
  this->LowColorButton->EnabledOff();

  this->HighColorLabel->SetParent(this->OutOfRangeFrame);
  this->HighColorLabel->SetText("High");
  this->HighColorLabel->Create();
  this->HighColorLabel->EnabledOff();

  this->HighColorButton->SetParent(this->OutOfRangeFrame);
  this->HighColorButton->Create();
  this->HighColorButton->LabelVisibilityOff();
  this->HighColorButton->SetColor(1, 1, 1);
  this->HighColorButton->SetCommand(this, "HighColorButtonCallback");
  this->HighColorButton->SetBalloonHelpString("Select the color for values above the high end of the scalar range.");
  this->HighColorButton->EnabledOff();

  this->Script("grid %s %s -sticky news %s",
               this->OutOfRangeLabel->GetWidgetName(),
               this->OutOfRangeCheck->GetWidgetName(), grid_settings);
  this->Script("grid %s -row 0 -column 2 -sticky e -padx 5 -pady 2",
               this->LowColorLabel->GetWidgetName());
  this->Script("grid %s -row 0 -column 3 -sticky w -padx 5 -pady 2",
               this->LowColorButton->GetWidgetName());
  this->Script("grid %s -row 0 -column 4 -sticky e -padx 5 -pady 2",
               this->HighColorLabel->GetWidgetName());
  this->Script("grid %s -row 0 -column 5 -sticky w -padx 5 -pady 2",
               this->HighColorButton->GetWidgetName());
  this->Script("grid columnconfigure %s 1 -weight 2",
               this->OutOfRangeFrame->GetWidgetName());
  this->Script("grid columnconfigure %s 2 -weight 2",
               this->OutOfRangeFrame->GetWidgetName());
  this->Script("grid columnconfigure %s 3 -weight 2",
               this->OutOfRangeFrame->GetWidgetName());
  this->Script("grid columnconfigure %s 4 -weight 2",
               this->OutOfRangeFrame->GetWidgetName());
  this->Script("grid columnconfigure %s 5 -weight 2",
               this->OutOfRangeFrame->GetWidgetName());

  // Scalar bar : Title control
  this->ScalarBarTitleFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->ScalarBarTitleFrame->Create();

  this->ScalarBarTitleLabel->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleLabel->SetText("Title:");
  this->ScalarBarTitleLabel->Create();
  this->ScalarBarTitleLabel->SetAnchorToWest();
  
  this->ScalarBarTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarTitleEntry->SetWidth(10);
  this->ScalarBarTitleEntry->Create();
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarTitleEntryCallback}",
               this->ScalarBarTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->ScalarBarVectorTitleEntry->SetParent(this->ScalarBarTitleFrame);
  this->ScalarBarVectorTitleEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth() / 2);
  this->ScalarBarVectorTitleEntry->Create();
  this->Script(
    "bind %s <KeyPress-Return> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarVectorTitleEntryCallback}",
               this->ScalarBarVectorTitleEntry->GetWidgetName(),
               this->GetTclName()); 

  this->TitleTextPropertyWidget->SetParent(this->ScalarBarTitleFrame);
  this->TitleTextPropertyWidget->Create();
  this->TitleTextPropertyWidget->SetChangedCommand(
    this, "TitleTextPropertyWidgetCallback");
  this->TitleTextPropertyWidget->GetTraceHelper()->SetReferenceHelper(
    this->GetTraceHelper());
  this->TitleTextPropertyWidget->GetTraceHelper()->SetReferenceCommand(
    "GetTitleTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarTitleLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarTitleEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -columnspan 2 -sticky nws %s",
               this->TitleTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : Label control
  this->ScalarBarLabelFormatFrame->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->ScalarBarLabelFormatFrame->Create();

  this->ScalarBarLabelFormatLabel->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatLabel->SetText("Labels:");
  this->ScalarBarLabelFormatLabel->Create();
  this->ScalarBarLabelFormatLabel->SetAnchorToWest();
  
  this->ScalarBarLabelFormatEntry->SetParent(this->ScalarBarLabelFormatFrame);
  this->ScalarBarLabelFormatEntry->SetWidth(
    this->ScalarBarTitleEntry->GetWidth());
  this->ScalarBarLabelFormatEntry->Create();
  this->Script("bind %s <KeyPress-Return> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName());
  this->Script("bind %s <FocusOut> {%s ScalarBarLabelFormatEntryCallback}",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               this->GetTclName()); 
  this->ScalarBarLabelFormatEntry->SetBalloonHelpString(
    "Set the labels format (printf() style).");

  this->LabelTextPropertyWidget->SetParent(this->ScalarBarLabelFormatFrame);

  // this will update this->LabelTextProperty
  this->LabelTextPropertyWidget->Create();
  this->LabelTextPropertyWidget->SetChangedCommand(this, 
    "LabelTextPropertyWidgetCallback");
  this->LabelTextPropertyWidget->GetTraceHelper()->SetReferenceHelper(
    this->GetTraceHelper());
  this->LabelTextPropertyWidget->GetTraceHelper()->SetReferenceCommand(
    "GetLabelTextPropertyWidget");

  this->Script("grid %s -row 0 -column 0 -sticky nws %s",
               this->ScalarBarLabelFormatLabel->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 0 -column 1 -sticky news %s",
               this->ScalarBarLabelFormatEntry->GetWidgetName(),
               grid_settings);

  this->Script("grid %s -row 1 -column 1 -sticky nws %s",
               this->LabelTextPropertyWidget->GetWidgetName(),
               grid_settings);

  // Scalar bar : enable copy between text property widgets
  this->TitleTextPropertyWidget->CopyVisibilityOn();
  this->TitleTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the labels text properties to the title text properties.");
  ostrstream copy1;
  copy1 << "CopyValuesFrom " << this->LabelTextPropertyWidget->GetTclName() << ends;
  this->TitleTextPropertyWidget->GetCopyButton()->SetCommand(
    this->TitleTextPropertyWidget, copy1.str());
  copy1.rdbuf()->freeze(0);

  this->LabelTextPropertyWidget->CopyVisibilityOn();
  this->LabelTextPropertyWidget->GetCopyButton()->SetBalloonHelpString(
    "Copy the title text properties to the labels text properties.");
  ostrstream copy2;
  copy2 << "CopyValuesFrom " << this->TitleTextPropertyWidget->GetTclName() << ends;
  this->LabelTextPropertyWidget->GetCopyButton()->SetCommand(
    this->LabelTextPropertyWidget, copy2.str());
  copy2.rdbuf()->freeze(0);

  // Scalar bar: synchronize all those grids to have them aligned
  const char *widgets[2];
  widgets[0] = this->ScalarBarTitleFrame->GetWidgetName();
  widgets[1] = this->ScalarBarLabelFormatFrame->GetWidgetName();

  int weights[2];
  weights[0] = 0;
  weights[1] = 4;

  float factors[2];
  factors[0] = 1.15;
  factors[1] = 0.0;

  vtkKWTkUtilities::SynchroniseGridsColumnMinimumSize(
    this->GetApplication()->GetMainInterp(), 2, widgets, factors, weights);

  // Number of labels thumb wheel
  this->NumberOfLabelsThumbWheel->SetParent(this->ScalarColorBarFrame->GetFrame());
  this->NumberOfLabelsThumbWheel->PopupModeOn();
  this->NumberOfLabelsThumbWheel->SetMinimumValue(2);
  this->NumberOfLabelsThumbWheel->ClampMinimumValueOn();
  this->NumberOfLabelsThumbWheel->Create();
  this->NumberOfLabelsThumbWheel->DisplayEntryOn();
  this->NumberOfLabelsThumbWheel->DisplayLabelOn();
  this->NumberOfLabelsThumbWheel->GetLabel()->SetText("Number of labels");
  this->NumberOfLabelsThumbWheel->SetValue(5);
  this->NumberOfLabelsThumbWheel->SetCommand(
    this, "NumberOfLabelsThumbWheelCallback");
  this->NumberOfLabelsThumbWheel->SetEndCommand(
    this, "NumberOfLabelsThumbWheelCallback");
  this->NumberOfLabelsThumbWheel->SetEntryCommand(
    this, "NumberOfLabelsThumbWheelCallback");

  // Back button
  this->BackButton->SetParent(this);
  this->BackButton->Create();
  this->BackButton->SetText("Back");
  this->BackButton->SetCommand(this, "BackButtonCallback");
  
  // Pack
  this->Script("pack %s %s -side top -expand t -anchor nw",
               this->VisibilityCheck->GetWidgetName(),
               this->DefaultCheck->GetWidgetName());
  this->Script("pack %s %s %s %s %s %s %s %s -side top -expand t -fill x -anchor nw",
               this->ParameterFrame->GetWidgetName(),
               this->ScalarRangeFrame->GetWidgetName(),
               this->ColorEditorFrame->GetWidgetName(),
               this->NumberOfColorsScale->GetWidgetName(),
               this->OutOfRangeFrame->GetWidgetName(),
               this->ScalarBarTitleFrame->GetWidgetName(),
               this->ScalarBarLabelFormatFrame->GetWidgetName(),
               this->NumberOfLabelsThumbWheel->GetWidgetName());

  this->Script("pack %s %s -side top -expand t -fill both -anchor w",
               this->ScalarBarTitleFrame->GetWidgetName(),
               this->ScalarBarLabelFormatFrame->GetWidgetName());

  this->Script("pack %s %s -side top -anchor n -fill x -padx 2 -pady 2",
               this->ScalarColorBarFrame->GetWidgetName(),
               this->BackButton->GetWidgetName());
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::VisibilityCheckCallback(int state)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->VisibilityCheck->SetSelectedState(state);
  this->CurrentColorMap->SetScalarBarVisibility(state);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::DefaultCheckCallback(int state)
{
  vtkKWApplication *app = this->GetApplication();
  app->SetRegistryValue(2, "RunTime", "UseColorMapDefaults", "%d", state);

  if (state)
    {
    app->SaveColorRegistryValue(2, "ColorMapStartColor",
                                this->StartColorButton->GetColor());
    app->SaveColorRegistryValue(2, "ColorMapEndColor",
                                this->EndColorButton->GetColor());
    app->SetRegistryValue(2, "RunTime", "ColorMapResolution", "%d",
                          (int)(this->NumberOfColorsScale->GetValue()));
    app->SetRegistryValue(2, "RunTime", "UseColorMapOutOfRange", "%d",
                          this->OutOfRangeCheck->GetSelectedState());
    app->SaveColorRegistryValue(2, "OutOfRangeLowColor",
                                this->LowColorButton->GetColor());
    app->SaveColorRegistryValue(2, "OutOfRangeHighColor",
                                this->HighColorButton->GetColor());
    vtkTextProperty *titleProp =
      this->TitleTextPropertyWidget->GetTextProperty();
    app->SaveColorRegistryValue(2, "ColorMapTitleColor",
                                titleProp->GetColor());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleFontFamily", "%d",
                          titleProp->GetFontFamily());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleBold", "%d",
                          titleProp->GetBold());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleItalic", "%d",
                          titleProp->GetItalic());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleShadow", "%d",
                          titleProp->GetShadow());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleOpacity", "%f",
                          titleProp->GetOpacity());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelFormat", "%s",
                          this->ScalarBarLabelFormatEntry->GetValue());
    vtkTextProperty *labelProp =
      this->LabelTextPropertyWidget->GetTextProperty();
    app->SaveColorRegistryValue(2, "ColorMapLabelColor",
                                labelProp->GetColor());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelFontFamily", "%d",
                          labelProp->GetFontFamily());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelBold", "%d",
                          labelProp->GetBold());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelItalic", "%d",
                          labelProp->GetItalic());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelShadow", "%d",
                          labelProp->GetShadow());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelOpacity", "%f",
                          labelProp->GetOpacity());
    app->SetRegistryValue(2, "RunTime", "ColorMapNumberOfLabels", "%d",
                          (int)(this->NumberOfLabelsThumbWheel->GetValue()));
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::VectorModeMagnitudeCallback()
{
  if (!this->CurrentColorMap)
    {
    return;
    }
  this->VectorComponentMenu->EnabledOff();
  this->CurrentColorMap->SetVectorMode(vtkLookupTable::MAGNITUDE);
  this->CurrentColorMap->ResetScalarRange();
  this->UpdateScalarBarTitle();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::VectorModeComponentCallback()
{
  if (!this->CurrentColorMap)
    {
    return;
    }
  this->VectorComponentMenu->EnabledOn();
  this->CurrentColorMap->SetVectorMode(vtkLookupTable::COMPONENT);
  this->CurrentColorMap->ResetScalarRange();
  this->UpdateVectorComponentMenu();
  this->UpdateScalarBarTitle();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::LockCheckCallback(int state)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetScalarRangeLock(state);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::ScalarRangeWidgetCallback(double, double)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  double range[2];
  this->ScalarRangeWidget->GetRange(range);
  this->CurrentColorMap->SetScalarRange(range[0], range[1]);
  int width = this->CurrentColorMap->GetMapWidth();
  int height = this->CurrentColorMap->GetMapHeight();
  this->CurrentColorMap->UpdateMap(width, height);
  this->UpdateOutOfRangeColors();
  this->Map->SetImageToPixels(this->CurrentColorMap->GetMapData(),
                              width, height, 3);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::StartColorButtonCallback(double r, double g, double b)
{
  if (!this->CurrentColorMap)
    {
    return;
    }
  this->CurrentColorMap->StartColorButtonCallback(r, g, b);
  int width = this->CurrentColorMap->GetMapWidth();
  int height = this->CurrentColorMap->GetMapHeight();
  this->CurrentColorMap->UpdateMap(width, height);
  this->Map->SetImageToPixels(this->CurrentColorMap->GetMapData(),
                              width, height, 3);
  if (this->DefaultCheck->GetSelectedState())
    {
    double rgb[3];
    rgb[0] = r;
    rgb[1] = g;
    rgb[2] = b;
    this->GetApplication()->SaveColorRegistryValue(2, "ColorMapStartColor",
                                                   rgb);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::MapConfigureCallback(int width, int height)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->UpdateMap(width, height);
  this->UpdateMapFromCurrentColorMap();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::EndColorButtonCallback(double r, double g, double b)
{
  if (!this->CurrentColorMap)
    {
    return;
    }
  this->CurrentColorMap->EndColorButtonCallback(r, g, b);
  int width = this->CurrentColorMap->GetMapWidth();
  int height = this->CurrentColorMap->GetMapHeight();
  this->CurrentColorMap->UpdateMap(width, height);
  this->Map->SetImageToPixels(this->CurrentColorMap->GetMapData(),
                              width, height, 3);
  if (this->DefaultCheck->GetSelectedState())
    {
    double rgb[3];
    rgb[0] = r;
    rgb[1] = g;
    rgb[2] = b;
    this->GetApplication()->SaveColorRegistryValue(2, "ColorMapEndColor",
                                                   rgb);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::SetColorSchemeToBlueRed()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetColorSchemeToBlueRed();
  this->StartColorButton->SetColor(this->CurrentColorMap->GetStartColor());
  this->EndColorButton->SetColor(this->CurrentColorMap->GetEndColor());
  
  this->UpdateMapFromCurrentColorMap();

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapStartColor", this->StartColorButton->GetColor());
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapEndColor", this->EndColorButton->GetColor());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::SetColorSchemeToRedBlue()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetColorSchemeToRedBlue();
  this->StartColorButton->SetColor(this->CurrentColorMap->GetStartColor());
  this->EndColorButton->SetColor(this->CurrentColorMap->GetEndColor());
  
  this->UpdateMapFromCurrentColorMap();

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapStartColor", this->StartColorButton->GetColor());
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapEndColor", this->EndColorButton->GetColor());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::SetColorSchemeToGrayscale()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetColorSchemeToGrayscale();
  this->StartColorButton->SetColor(this->CurrentColorMap->GetStartColor());
  this->EndColorButton->SetColor(this->CurrentColorMap->GetEndColor());
  
  this->UpdateMapFromCurrentColorMap();

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapStartColor", this->StartColorButton->GetColor());
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapEndColor", this->EndColorButton->GetColor());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::SetColorSchemeToLabBlueRed()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetColorSchemeToLabBlueRed();
  this->StartColorButton->SetColor(this->CurrentColorMap->GetStartColor());
  this->EndColorButton->SetColor(this->CurrentColorMap->GetEndColor());

  this->UpdateMapFromCurrentColorMap();

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapStartColor", this->StartColorButton->GetColor());
    this->GetApplication()->SaveColorRegistryValue(
      2, "ColorMapEndColor", this->EndColorButton->GetColor());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::NumberOfColorsScaleCallback(double value)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetNumberOfColors((int)value);
  this->UpdateMapFromCurrentColorMap();
  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SetRegistryValue(2, "RunTime",
                                             "ColorMapResolution", "%d",
                                             value);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::OutOfRangeCheckCallback(int state)
{
  if (!state)
    {
    this->LowColorLabel->EnabledOff();
    this->LowColorButton->EnabledOff();
    this->HighColorLabel->EnabledOff();
    this->HighColorButton->EnabledOff();
    if (this->CurrentColorMap)
      {
      this->CurrentColorMap->SetUseLowOutOfRangeColor(0);
      this->CurrentColorMap->SetUseHighOutOfRangeColor(0);
      this->CurrentColorMap->RenderView();
      this->MapConfigureCallback(this->CurrentColorMap->GetMapWidth(),
                                 this->CurrentColorMap->GetMapHeight());
      }
    }
  else
    {
    this->LowColorLabel->EnabledOn();
    this->LowColorButton->EnabledOn();
    this->HighColorLabel->EnabledOn();
    this->HighColorButton->EnabledOn();
    this->UpdateOutOfRangeColors();
    }
  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SetRegistryValue(2, "RunTime",
                                             "UseColorMapOutOfRange", "%d",
                                             state);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::LowColorButtonCallback(double r, double g, double b)
{
  this->UpdateOutOfRangeColors();
  if (this->DefaultCheck->GetSelectedState())
    {
    double rgb[3];
    rgb[0] = r;
    rgb[1] = g;
    rgb[2] = b;
    this->GetApplication()->SaveColorRegistryValue(2, "OutOfRangeLowColor",
                                                   rgb);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::HighColorButtonCallback(double r, double g, double b)
{
  this->UpdateOutOfRangeColors();
  if (this->DefaultCheck->GetSelectedState())
    {
    double rgb[3];
    rgb[0] = r;
    rgb[1] = g;
    rgb[2] = b;
    this->GetApplication()->SaveColorRegistryValue(2, "OutOfRangeHighColor",
                                                   rgb);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::ScalarBarTitleEntryCallback()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetScalarBarTitle(
    this->ScalarBarTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::ScalarBarVectorTitleEntryCallback()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetScalarBarVectorTitle(
    this->ScalarBarVectorTitleEntry->GetValue());
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::TitleTextPropertyWidgetCallback()
{
  vtkTextProperty *prop = this->TitleTextPropertyWidget->GetTextProperty();
  if (!prop || !this->CurrentColorMap)
    {
    return;
    }

  double *color = prop->GetColor();
  this->CurrentColorMap->SetTitleColor(color[0], color[1], color[2]);
  this->CurrentColorMap->SetTitleOpacity(prop->GetOpacity());
  this->CurrentColorMap->SetTitleFontFamily(prop->GetFontFamily());
  this->CurrentColorMap->SetTitleBold(prop->GetBold());
  this->CurrentColorMap->SetTitleItalic(prop->GetItalic());
  this->CurrentColorMap->SetTitleShadow(prop->GetShadow());

  if (this->DefaultCheck->GetSelectedState())
    {
    vtkKWApplication *app = this->GetApplication();
    app->SaveColorRegistryValue(2, "ColorMapTitleColor", prop->GetColor());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleFontFamily", "%d",
                          prop->GetFontFamily());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleBold", "%d",
                          prop->GetBold());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleItalic", "%d",
                          prop->GetItalic());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleShadow", "%d",
                          prop->GetShadow());
    app->SetRegistryValue(2, "RunTime", "ColorMapTitleOpacity", "%f",
                          prop->GetOpacity());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::ScalarBarLabelFormatEntryCallback()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetScalarBarLabelFormat(
    this->ScalarBarLabelFormatEntry->GetValue());

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SetRegistryValue(
      2, "RunTime", "ColorMapLabelFormat", "%s",
      this->ScalarBarLabelFormatEntry->GetValue());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::LabelTextPropertyWidgetCallback()
{
  vtkTextProperty *prop = this->LabelTextPropertyWidget->GetTextProperty();
  if (!prop || !this->CurrentColorMap)
    {
    return;
    }

  double *color = prop->GetColor();
  this->CurrentColorMap->SetLabelColor(color[0], color[1], color[2]);
  this->CurrentColorMap->SetLabelOpacity(prop->GetOpacity());
  this->CurrentColorMap->SetLabelFontFamily(prop->GetFontFamily());
  this->CurrentColorMap->SetLabelBold(prop->GetBold());
  this->CurrentColorMap->SetLabelItalic(prop->GetItalic());
  this->CurrentColorMap->SetLabelShadow(prop->GetShadow());

  if (this->DefaultCheck->GetSelectedState())
    {
    vtkKWApplication *app = this->GetApplication();
    app->SaveColorRegistryValue(2, "ColorMapLabelColor", prop->GetColor());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelFontFamily", "%d",
                          prop->GetFontFamily());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelBold", "%d",
                          prop->GetBold());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelItalic", "%d",
                          prop->GetItalic());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelShadow", "%d",
                          prop->GetShadow());
    app->SetRegistryValue(2, "RunTime", "ColorMapLabelOpacity", "%f",
                          prop->GetOpacity());
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateParameterList(vtkPVWindow *win)
{
  vtkPVSourceCollection *sources = win->GetSourceList("Sources");
  sources->InitTraversal();
  vtkPVSource *source = NULL;
  vtkSMDataObjectDisplayProxy *dProxy;
  vtkPVGeometryInformation *geomInfo;
  vtkPVDataSetAttributesInformation *attrInfo;
  vtkPVArrayInformation *arrayInfo;
  int i, numArrays, numComp;
  int arraysFound = 0;

  this->ParameterMenu->GetMenu()->DeleteAllItems();

  while ((source = sources->GetNextPVSource()))
    {
    dProxy = source->GetDisplayProxy();
    geomInfo = dProxy->GetGeometryInformation();
    attrInfo = geomInfo->GetPointDataInformation();

    numArrays = attrInfo->GetNumberOfArrays();
    arraysFound += numArrays;

    for (i = 0; i < numArrays; i++)
      {
      arrayInfo = attrInfo->GetArrayInformation(i);
      ostrstream label;
      label << "Point " << arrayInfo->GetName();
      numComp = arrayInfo->GetNumberOfComponents();
      if (numComp > 1)
        {
        label << " (" << numComp << ")";
        }
      label << ends;
      if (this->ParameterMenu->GetMenu()->GetIndexOfItem(label.str()) < 0)
        {
        ostrstream method;
        method << "UpdateColorMapUI " << arrayInfo->GetName() << " "
               << arrayInfo->GetNumberOfComponents() << " "
               << vtkSMDataObjectDisplayProxy::POINT_FIELD_DATA << ends;
        this->ParameterMenu->GetMenu()->AddRadioButton(
          label.str(), this, method.str());
        method.rdbuf()->freeze(0);
        }
      label.rdbuf()->freeze(0);
      }

    attrInfo = geomInfo->GetCellDataInformation();
    numArrays = attrInfo->GetNumberOfArrays();
    arraysFound += numArrays;

    for (i = 0; i < numArrays; i++)
      {
      arrayInfo = attrInfo->GetArrayInformation(i);
      ostrstream label;
      label << "Cell " << arrayInfo->GetName();
      numComp = arrayInfo->GetNumberOfComponents();
      if (numComp > 1)
        {
        label << " (" << numComp << ")";
        }
      label << ends;
      if (this->ParameterMenu->GetMenu()->GetIndexOfItem(label.str()) < 0)
        {
        ostrstream method;
        method << "UpdateColorMapUI " << arrayInfo->GetName() << " "
               << arrayInfo->GetNumberOfComponents() << " "
               << vtkSMDataObjectDisplayProxy::CELL_FIELD_DATA << ends;
        this->ParameterMenu->GetMenu()->AddRadioButton(
          label.str(), this, method.str());
        method.rdbuf()->freeze(0);
        }
      label.rdbuf()->freeze(0);
      }
    }

  if (!arraysFound)
    {
    this->VisibilityCheck->EnabledOff();
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateColorMapUI(const char* name, int numComponents,
                                       int field)
{
  vtkPVApplication *pvApp = vtkPVApplication::SafeDownCast(
    this->GetApplication());
  vtkPVWindow *win = pvApp->GetMainWindow();
  vtkPVColorMap *colorMap = win->GetPVColorMap(name, numComponents);

  ostrstream label;
  if (field == vtkSMDataObjectDisplayProxy::POINT_FIELD_DATA)
    {
    label << "Point ";
    }
  else
    {
    label << "Cell ";
    }
  label << name;
  if (numComponents > 1)
    {
    label << " (" << numComponents << ")";
    }
  label << ends;
  this->ParameterMenu->SetValue(label.str());
  label.rdbuf()->freeze(0);

  vtkPVSourceCollection *sources = win->GetSourceList("Sources");
  sources->InitTraversal();
  vtkPVSource *src;
  int visible = 0;

  while ((src = sources->GetNextPVSource()))
    {
    if (src->GetPVColorMap() == colorMap)
      {
      visible = 1;
      break;
      }
    }

  this->SetCurrentColorMap(colorMap);
  this->VisibilityCheck->SetEnabled(visible);
  this->DefaultCheck->SetSelectedState(0);

  this->VisibilityCheck->SetSelectedState(colorMap->GetScalarBarVisibility());
  if (colorMap->GetNumberOfVectorComponents() > 1)
    {
    this->VectorLabel->EnabledOn();
    this->VectorModeMenu->EnabledOn();
    this->VectorModeMenu->SetValue(
      this->ConvertVectorModeToString(colorMap->GetVectorMode()));
    this->Script("grid %s -row 0 -column 2 -sticky news -padx 1 -pady 2",
                 this->ScalarBarVectorTitleEntry->GetWidgetName());
    this->Script("grid columnconfigure %s 2 -weight 3",
                 this->ScalarBarTitleFrame->GetWidgetName());
    if (!strcmp(this->ConvertVectorModeToString(colorMap->GetVectorMode()),
                "Component"))
      {
      this->VectorComponentMenu->EnabledOn();
      this->UpdateVectorComponentMenu();
      }
    else
      {
      this->ScalarBarVectorTitleEntry->SetValue(
        colorMap->GetVectorMagnitudeTitle());
      }
    }
  else
    {
    this->VectorLabel->EnabledOff();
    this->VectorModeMenu->EnabledOff();
    this->VectorComponentMenu->EnabledOff();
    this->Script("catch {eval grid forget %s}",
                 this->ScalarBarVectorTitleEntry->GetWidgetName());
    this->Script("grid columnconfigure %s 2 -weight 0",
                 this->ScalarBarTitleFrame->GetWidgetName());
    }
  this->ScalarRangeLockCheck->SetSelectedState(colorMap->GetScalarRangeLock());
  this->ScalarRangeWidget->SetWholeRange(colorMap->GetWholeScalarRange());
  this->ScalarRangeWidget->SetRange(colorMap->GetScalarRange());
  this->StartColorButton->SetColor(colorMap->GetStartColor());
  this->EndColorButton->SetColor(colorMap->GetEndColor());
  int oldHeight = this->Map->GetHeight();
  int oldWidth = this->Map->GetWidth();
  int height = colorMap->GetMapHeight();
  int width = colorMap->GetMapWidth();
  this->Map->SetHeight(height);
  this->Map->SetWidth(width);
  this->Map->SetImageToPixels(colorMap->GetMapData(), width, height, 3);

  this->NumberOfColorsScale->SetValue(colorMap->GetNumberOfColors());
  this->ScalarBarTitleEntry->SetValue(colorMap->GetScalarBarTitle());
  this->ScalarBarLabelFormatEntry->SetValue(
    colorMap->GetScalarBarLabelFormat());
  this->TitleTextPropertyWidget->SetTextProperty(
    colorMap->GetTitleTextProperty());
  this->TitleTextPropertyWidget->Update();
  this->LabelTextPropertyWidget->SetTextProperty(
    colorMap->GetLabelTextProperty());
  this->LabelTextPropertyWidget->Update();

  int outOfRange = 0;
  if (colorMap->GetUseLowOutOfRangeColor() ||
      colorMap->GetUseHighOutOfRangeColor())
    {
    outOfRange = 1;
    }
  
  this->OutOfRangeCheck->SetSelectedState(outOfRange);
  this->LowColorButton->SetColor(colorMap->GetLowLookupTableValue());
  this->HighColorButton->SetColor(colorMap->GetHighLookupTableValue());
  
  vtkKWApplication *app = this->GetApplication();
  if (app->GetIntRegistryValue(2, "RunTime", "UseColorMapDefaults") &&
      !colorMap->GetDisplayed())
    {
    double rgb[3];
    app->RetrieveColorRegistryValue(2, "ColorMapStartColor", rgb);
    this->StartColorButton->SetColor(rgb);
    this->CurrentColorMap->StartColorButtonCallback(rgb[0], rgb[1], rgb[2]);
    app->RetrieveColorRegistryValue(2, "ColorMapEndColor", rgb);
    this->EndColorButton->SetColor(rgb);
    this->CurrentColorMap->EndColorButtonCallback(rgb[0], rgb[1], rgb[2]);
    this->CurrentColorMap->UpdateMap(width, height);
    this->Map->SetImageToPixels(colorMap->GetMapData(), width, height, 3);
    this->NumberOfColorsScale->SetValue(app->GetIntRegistryValue(
                                          2, "RunTime", "ColorMapResolution"));
    this->OutOfRangeCheck->SetSelectedState(app->GetIntRegistryValue(
                                              2, "RunTime",
                                              "UseColorMapOutOfRange"));
    if (this->OutOfRangeCheck->GetSelectedState())
      {
      app->RetrieveColorRegistryValue(2, "OutOfRangeLowColor", rgb);
      this->LowColorButton->SetColor(rgb);
      app->RetrieveColorRegistryValue(2, "OutOfRangeHighColor", rgb);
      this->HighColorButton->SetColor(rgb);
      }

    app->RetrieveColorRegistryValue(2, "ColorMapTitleColor", rgb);
    this->TitleTextPropertyWidget->SetColor(rgb);
    this->TitleTextPropertyWidget->SetFontFamily(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapTitleFontFamily"));
    this->TitleTextPropertyWidget->SetBold(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapTitleBold"));
    this->TitleTextPropertyWidget->SetItalic(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapTitleItalic"));
    this->TitleTextPropertyWidget->SetShadow(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapTitleShadow"));
    this->TitleTextPropertyWidget->SetOpacity(
      app->GetFloatRegistryValue(2, "RunTime", "ColorMapTitleOpacity"));
    char labelFormat[100];
    app->GetRegistryValue(2, "RunTime", "ColorMapLabelFormat", labelFormat);
    this->ScalarBarLabelFormatEntry->SetValue(labelFormat);
    app->RetrieveColorRegistryValue(2, "ColorMapLabelColor", rgb);
    this->LabelTextPropertyWidget->SetColor(rgb);
    this->LabelTextPropertyWidget->SetFontFamily(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapLabelFontFamily"));
    this->LabelTextPropertyWidget->SetBold(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapLabelBold"));
    this->LabelTextPropertyWidget->SetItalic(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapLabelItalic"));
    this->LabelTextPropertyWidget->SetShadow(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapLabelShadow"));
    this->LabelTextPropertyWidget->SetOpacity(
      app->GetFloatRegistryValue(2, "RunTime", "ColorMapLabelOpacity"));
    this->NumberOfLabelsThumbWheel->SetValue(
      app->GetIntRegistryValue(2, "RunTime", "ColorMapNumberOfLabels"));
    }

  if (oldWidth > MAP_WIDTH_INITIAL || oldHeight > MAP_HEIGHT_INITIAL)
    {
    this->MapConfigureCallback(oldWidth, oldHeight);
    }

  colorMap->SetDisplayed(1);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateMapFromCurrentColorMap()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  int width = this->CurrentColorMap->GetMapWidth();
  int height = this->CurrentColorMap->GetMapHeight();
  this->Map->SetWidth(width);
  this->Map->SetHeight(height);
  this->Map->SetImageToPixels(this->CurrentColorMap->GetMapData(),
                              width, height, 3);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::VectorComponentMenuCallback(int idx)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetVectorComponent(idx);
  this->UpdateScalarBarTitle();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateScalarBarTitle()
{
  if (this->CurrentColorMap == NULL ||
      this->CurrentColorMap->GetScalarBarTitle() == NULL ||
      this->GetApplication() == NULL)
    {
    return;
    }
  
  if (this->CurrentColorMap->GetVectorMode() == vtkLookupTable::MAGNITUDE &&
      this->CurrentColorMap->GetNumberOfVectorComponents() > 1)
    {
    ostrstream ostr;
    ostr << this->CurrentColorMap->GetScalarBarTitle() << " "
         << this->CurrentColorMap->GetVectorMagnitudeTitle() << ends;
    this->CurrentColorMap->SetTitle(ostr.str()); 
    ostr.rdbuf()->freeze(0);    
    this->ScalarBarVectorTitleEntry->SetValue(
      this->CurrentColorMap->GetVectorMagnitudeTitle());
    }
  else if (this->CurrentColorMap->GetNumberOfVectorComponents() == 1)
    {
    this->CurrentColorMap->SetTitle(
      this->CurrentColorMap->GetScalarBarTitle());
    }
  else
    {
    ostrstream ostr;
    ostr << this->CurrentColorMap->GetScalarBarTitle() << " " 
         << this->CurrentColorMap->GetVectorComponentTitle(
           this->CurrentColorMap->GetVectorComponent())
         << ends;
    this->CurrentColorMap->SetTitle(ostr.str());
    ostr.rdbuf()->freeze(0);    
    this->ScalarBarVectorTitleEntry->SetValue(
      this->CurrentColorMap->GetVectorComponentTitle(
        this->CurrentColorMap->GetVectorComponent()));
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateVectorComponentMenu()
{
  int numComps = this->CurrentColorMap->GetNumberOfVectorComponents();
  int comp = this->CurrentColorMap->GetVectorComponent();
  this->VectorComponentMenu->GetMenu()->DeleteAllItems();

  int i;
  for (i = 0; i < numComps; i++)
    {
    ostrstream cmd;
    cmd << "VectorComponentMenuCallback " << i << ends;
    this->VectorComponentMenu->GetMenu()->AddRadioButton(
      this->CurrentColorMap->GetVectorComponentTitle(i), this, cmd.str());
    cmd.rdbuf()->freeze(0);
    }
  this->VectorComponentMenu->SetValue(
    this->CurrentColorMap->GetVectorComponentTitle(comp));
  this->ScalarBarVectorTitleEntry->SetValue(
    this->CurrentColorMap->GetVectorComponentTitle(comp));
}

//----------------------------------------------------------------------------
const char* vtkPVColorMapUI::ConvertVectorModeToString(int mode)
{
  switch (mode)
    {
    case vtkLookupTable::MAGNITUDE:
      return "Magnitude";
    case vtkLookupTable::COMPONENT:
      return "Component";
    default:
      return NULL;
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateScalarRangeWidget()
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->ScalarRangeWidget->SetWholeRange(
    this->CurrentColorMap->GetWholeScalarRange());
  this->ScalarRangeWidget->SetRange(this->CurrentColorMap->GetScalarRange());
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateEnableState()
{
  this->Superclass::UpdateEnableState();

  this->PropagateEnableState(this->ScalarColorBarFrame);
  this->PropagateEnableState(this->VisibilityCheck);
  this->PropagateEnableState(this->DefaultCheck);
  this->PropagateEnableState(this->ParameterFrame);
  this->PropagateEnableState(this->ParameterLabel);
  this->PropagateEnableState(this->ParameterMenu);
  this->PropagateEnableState(this->VectorLabel);
  this->PropagateEnableState(this->VectorModeMenu);
  this->PropagateEnableState(this->VectorComponentMenu);
  this->PropagateEnableState(this->ScalarRangeFrame);
  this->PropagateEnableState(this->ScalarRangeLockCheck);
  this->PropagateEnableState(this->ScalarRangeWidget);
  this->PropagateEnableState(this->NumberOfColorsScale);
  this->PropagateEnableState(this->ColorEditorFrame);
  this->PropagateEnableState(this->PresetsMenuButton);
  this->PropagateEnableState(this->StartColorButton);
  this->PropagateEnableState(this->Map);
  this->PropagateEnableState(this->EndColorButton);
  this->PropagateEnableState(this->OutOfRangeFrame);
  this->PropagateEnableState(this->OutOfRangeLabel);
  this->PropagateEnableState(this->OutOfRangeCheck);
  this->PropagateEnableState(this->LowColorLabel);
  this->PropagateEnableState(this->LowColorButton);
  this->PropagateEnableState(this->HighColorLabel);
  this->PropagateEnableState(this->HighColorButton);
  this->PropagateEnableState(this->ScalarBarTitleFrame);
  this->PropagateEnableState(this->ScalarBarTitleLabel);
  this->PropagateEnableState(this->ScalarBarTitleEntry);
  this->PropagateEnableState(this->ScalarBarVectorTitleEntry);
  this->PropagateEnableState(this->TitleTextPropertyWidget);
  this->PropagateEnableState(this->ScalarBarLabelFormatFrame);
  this->PropagateEnableState(this->ScalarBarLabelFormatLabel);
  this->PropagateEnableState(this->ScalarBarLabelFormatEntry);
  this->PropagateEnableState(this->LabelTextPropertyWidget);
  this->PropagateEnableState(this->NumberOfLabelsThumbWheel);
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::NumberOfLabelsThumbWheelCallback(int num)
{
  if (!this->CurrentColorMap)
    {
    return;
    }

  this->CurrentColorMap->SetNumberOfLabels(num);

  if (this->DefaultCheck->GetSelectedState())
    {
    this->GetApplication()->SetRegistryValue(2, "RunTime",
                                             "ColorMapNumberOfLabels", "%d",
                                             num);
    }
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::BackButtonCallback()
{
  vtkPVApplication *pvApp = vtkPVApplication::SafeDownCast(
    this->GetApplication());
  vtkPVWindow *win = pvApp->GetMainWindow();
  win->ShowCurrentSourceProperties();
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::UpdateOutOfRangeColors()
{
  if (!this->CurrentColorMap || !this->OutOfRangeCheck->GetSelectedState())
    {
    return;
    }

  double *lowColor, *highColor;
  lowColor = this->LowColorButton->GetColor();
  highColor = this->HighColorButton->GetColor();
  double range[2], wholeRange[2];
  this->ScalarRangeWidget->GetRange(range);
  if (!this->CurrentColorMap->ComputeWholeScalarRange(wholeRange))
    {
    return;
    }

  if (range[0] > wholeRange[0])
    {
    this->CurrentColorMap->SetLowLookupTableValue(lowColor);
    this->CurrentColorMap->SetUseLowOutOfRangeColor(1);
    }
  else
    {
    this->CurrentColorMap->SetUseLowOutOfRangeColor(0);
    }
  if (range[1] < wholeRange[1])
    {
    this->CurrentColorMap->SetHighLookupTableValue(highColor);
    this->CurrentColorMap->SetUseHighOutOfRangeColor(1);
    }
  else
    {
    this->CurrentColorMap->SetUseHighOutOfRangeColor(0);
    }

  this->CurrentColorMap->RenderView();
  this->MapConfigureCallback(this->CurrentColorMap->GetMapWidth(),
                             this->CurrentColorMap->GetMapHeight());
}

//----------------------------------------------------------------------------
void vtkPVColorMapUI::PrintSelf(ostream &os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  os << indent << "TitleTextPropertyWidget: " << this->TitleTextPropertyWidget
     << endl;
  os << indent << "LabelTextPropertyWidget: " << this->LabelTextPropertyWidget
     << endl;
  os << indent << "CurrentColorMap: " << this->CurrentColorMap << endl;
}
