/**
 * Simple C++ program which exercises basic pointer size, creating and joining
 * threads.
 */
#include <cstdlib>
#include <iostream>
#include <thread>
#include <vector>
#include <vtkSMPTools.h>
#include <vtkSmartPointer.h>
#include <vtkTypeUInt32Array.h>

int main(int argc, char *argv[]) {
  if (argc < 3) {
    std::cerr << "Usage: " << argv[0] << " <wordsize> <num_threads>\n";
    return 1;
  }
  std::size_t wordsize = std::atoi(argv[1]);
  if ((wordsize == 32 && sizeof(void *) != 4) ||
      (wordsize == 64 && sizeof(void *) != 8)) {
    std::cerr << "Test failed for word size=" << wordsize << '\n';
    return EXIT_FAILURE;
  }
  if (wordsize != 32 && wordsize != 64) {
    std::cerr << "Word size is unsupported " << wordsize << '\n';
    return EXIT_FAILURE;
  }

  std::size_t numThreads = std::atoi(argv[2]);
  if (numThreads > 0) {
    auto sharedData = vtk::TakeSmartPointer(vtkTypeUInt32Array::New());
    sharedData->SetNumberOfValues(numThreads);
    // Test native C++ threads
    std::vector<std::thread> threads;
    for (std::size_t i = 0; i < numThreads; ++i)
      threads.emplace_back(
          [i, sharedData]() { sharedData->SetTypedComponent(i, 0, 2 * i); });
    for (auto &t : threads)
      t.join();
    for (std::size_t i = 0; i < numThreads; ++i)
      if (sharedData->GetTypedComponent(i, 0) != 2 * i) {
        std::cerr << "Incorrect value at threadId=" << i << '\n';
        return EXIT_FAILURE;
      }
    // Test VTK SMP tools
    vtkSMPTools::SetBackend("STDThread");
    vtkSMPTools::Initialize();
    vtkSMPTools::For(0, numThreads,
                     [sharedData](vtkIdType iStart, vtkIdType iEnd) {
                       for (vtkIdType i = iStart; i < iEnd; ++i) {
                         sharedData->SetTypedComponent(i, 0, 3 * i);
                       }
                     });
    for (std::size_t i = 0; i < numThreads; ++i)
      if (sharedData->GetTypedComponent(i, 0) != 3 * i) {
        std::cerr << "Incorrect value at threadId=" << i << '\n';
        return EXIT_FAILURE;
      }
  }
  return EXIT_SUCCESS;
}
