#!/bin/sh
set -ex

script_dir="$(cd "$(dirname "$0")"; pwd)"
cd "$script_dir"

# Set VTK_VCS_REF to 'master' if not already set (POSIX-compliant defaulting)
: "${VTK_VCS_REF:=master}"
# Set VTK_VCS_URL to the default URL if not already set
: "${VTK_VCS_URL:=https://gitlab.kitware.com/vtk/vtk.git}"

case "$VTK_BUILD_ARCHITECTURE" in
    wasm32)          cmake_configuration=wasm32_emscripten_linux; enable_64_bit=OFF; enable_serialization=ON;  enable_threads=OFF ;;
    wasm32-threads)  cmake_configuration=wasm32_emscripten_linux; enable_64_bit=OFF; enable_serialization=OFF; enable_threads=ON  ;;
    wasm64)          cmake_configuration=wasm64_emscripten_linux; enable_64_bit=ON;  enable_serialization=ON;  enable_threads=OFF ;;
    wasm64-threads)  cmake_configuration=wasm64_emscripten_linux; enable_64_bit=ON;  enable_serialization=OFF; enable_threads=ON  ;;
    *) echo "Unrecognized build architecture $VTK_BUILD_ARCHITECTURE"; exit 1 ;;
esac

: "${JOB_POOL_LINK_SIZE:=$(nproc)}"

[ -d .vtk ] || git clone "$VTK_VCS_URL" .vtk
cd .vtk
git checkout "$VTK_VCS_REF"
git submodule update --init --recursive
short_sha="$(git rev-parse --short HEAD)"
parse_version() {
    [ -f "$1" ] && sed -n 's/.*set(VTK_\(MAJOR\|MINOR\|BUILD\)_VERSION \([0-9]\+\)).*/\2/p' "$1" | paste -sd. - || echo "unknown.unknown.unknown"
}
version=$(parse_version CMake/vtkVersion.cmake)
./.gitlab/ci/sccache.sh
cd ..

docker=podman
image_tag="$VTK_BUILD_ARCHITECTURE"

$docker build --format=docker \
    --cgroup-manager=cgroupfs \
    --volume "$script_dir/.vtk:/VTK-src:Z" \
    --build-arg SCCACHE_REDIS="$SCCACHE_REDIS" \
    --build-arg CMAKE_CONFIGURATION="$cmake_configuration" \
    --build-arg ENABLE_64_BIT="$enable_64_bit" \
    --build-arg ENABLE_SERIALIZATION="$enable_serialization" \
    --build-arg ENABLE_THREADS="$enable_threads" \
    --build-arg JOB_POOL_LINK_SIZE="$JOB_POOL_LINK_SIZE" \
    -t "kitware/vtk-wasm-sdk:$image_tag" \
    -f "$script_dir/Dockerfile.build" "$@" | tee "$script_dir/build.log"

$docker save --format=docker-archive -o "$script_dir/vtk-wasm-sdk-emscripten-$image_tag.tar" "kitware/vtk-wasm-sdk:$image_tag"
echo "$version" > "$script_dir/$VTK_BUILD_ARCHITECTURE.version"
echo "$short_sha" > "$script_dir/$VTK_BUILD_ARCHITECTURE.commit"
