#!/usr/bin/env python

import vtk


def main():
    xyzFn, qFn = get_program_parameters()
    colors = vtk.vtkNamedColors()

    pl3d = vtk.vtkMultiBlockPLOT3DReader()
    pl3d.SetXYZFileName(xyzFn)
    pl3d.SetQFileName(qFn)
    pl3d.SetScalarFunctionNumber(100)
    pl3d.SetVectorFunctionNumber(202)
    pl3d.Update()

    pl3dOutput = pl3d.GetOutput().GetBlock(0)

    plane = vtk.vtkStructuredGridGeometryFilter()
    plane.SetInputData(pl3dOutput)
    plane.SetExtent(1, 100, 1, 100, 7, 7)

    lut = vtk.vtkLookupTable()

    planeMapper = vtk.vtkPolyDataMapper()
    planeMapper.SetLookupTable(lut)
    planeMapper.SetInputConnection(plane.GetOutputPort())
    planeMapper.SetScalarRange(pl3dOutput.GetScalarRange())

    planeActor = vtk.vtkActor()
    planeActor.SetMapper(planeMapper)

    # This creates an outline around the data.

    outline = vtk.vtkStructuredGridOutlineFilter()
    outline.SetInputData(pl3dOutput)

    outlineMapper = vtk.vtkPolyDataMapper()
    outlineMapper.SetInputConnection(outline.GetOutputPort())

    outlineActor = vtk.vtkActor()
    outlineActor.SetMapper(outlineMapper)

    # Much of the following is commented out. To try different lookup tables,
    # uncomment the appropriate portions.
    #

    # This creates a black to white lut.
    # lut.SetHueRange(0, 0)
    # lut.SetSaturationRange(0, 0)
    # lut.SetValueRange(0.2, 1.0)

    # This creates a red to blue lut.
    # lut.SetHueRange(0.0, 0.667)

    # This creates a blue to red lut.
    # lut.SetHueRange(0.667, 0.0)

    # This creates a weird effect. The Build() method causes the lookup table
    # to allocate memory and create a table based on the correct hue, saturation,
    # value, and alpha (transparency) range. Here we then manually overwrite the
    # values generated by the Build() method.
    lut.SetNumberOfColors(256)
    lut.SetHueRange(0.0, 0.667)
    lut.Build()

    # Create the RenderWindow, Renderer and both Actors.
    #
    ren1 = vtk.vtkRenderer()
    renWin = vtk.vtkRenderWindow()
    renWin.AddRenderer(ren1)
    iren = vtk.vtkRenderWindowInteractor()
    iren.SetRenderWindow(renWin)

    # Add the actors to the renderer, set the background and size.
    #
    ren1.AddActor(outlineActor)
    ren1.AddActor(planeActor)

    ren1.SetBackground(colors.GetColor3d('SlateGray'))
    ren1.TwoSidedLightingOff()

    renWin.SetSize(512, 512)
    renWin.SetWindowName('Rainbow')

    iren.Initialize()

    cam1 = ren1.GetActiveCamera()
    cam1.SetClippingRange(3.95297, 50)
    cam1.SetFocalPoint(8.88908, 0.595038, 29.3342)
    cam1.SetPosition(-12.3332, 31.7479, 41.2387)
    cam1.SetViewUp(0.060772, -0.319905, 0.945498)

    iren.Start()


def get_program_parameters():
    import argparse
    description = 'Demonstrates the use and manipulation of lookup tables.'
    epilogue = '''
    First create a simple pipeline that reads a structured grid
    and then extracts a plane from the grid. The plane will be colored
    differently by using different lookup tables.
    
    Note: The Update method is manually invoked because it causes the
    reader to read; later on we use the output of the reader to set
    a range for the scalar values.
    '''
    parser = argparse.ArgumentParser(description=description, epilog=epilogue,
                                     formatter_class=argparse.RawDescriptionHelpFormatter)
    parser.add_argument('xyzFilename', help='combxyz.bin.')
    parser.add_argument('qFilename', help='combq.bin.')
    args = parser.parse_args()
    return args.xyzFilename, args.qFilename


if __name__ == '__main__':
    main()
