#!/usr/bin/env python3

from dataclasses import dataclass

# noinspection PyUnresolvedReferences
import vtkmodules.vtkInteractionStyle
# noinspection PyUnresolvedReferences
import vtkmodules.vtkRenderingOpenGL2
from vtkmodules.vtkCommonColor import vtkNamedColors
from vtkmodules.vtkCommonCore import vtkPoints
from vtkmodules.vtkCommonDataModel import (
    vtkCellArray,
    vtkLine,
    vtkPolyData
)
from vtkmodules.vtkFiltersModeling import vtkRuledSurfaceFilter
from vtkmodules.vtkRenderingCore import (
    vtkActor,
    vtkPolyDataMapper,
    vtkRenderWindow,
    vtkRenderWindowInteractor,
    vtkRenderer
)


def main():
    colors = vtkNamedColors()

    # Create a renderer, rendering window and interactor.
    ren = vtkRenderer(background=colors.GetColor3d('SteelBlue'))
    ren_win = vtkRenderWindow(window_name='RuledSurfaceFilter')
    ren_win.AddRenderer(ren)
    iren = vtkRenderWindowInteractor()
    iren.render_window = ren_win
    # Create the points for the lines.
    points = vtkPoints()
    points.InsertPoint(0, 0, 0, 1)
    points.InsertPoint(1, 1, 0, 0)
    points.InsertPoint(2, 0, 1, 0)
    points.InsertPoint(3, 1, 1, 1)

    # Create line1.
    line1 = vtkLine()
    line1.GetPointIds().SetId(0, 0)
    line1.GetPointIds().SetId(1, 1)

    # Create line2.
    line2 = vtkLine()
    line2.GetPointIds().SetId(0, 2)
    line2.GetPointIds().SetId(1, 3)

    # Create a cellArray containing the lines.
    lines = vtkCellArray()
    lines.InsertNextCell(line1)
    lines.InsertNextCell(line2)

    # Create the vtkPolyData to contain the points and cellArray with the lines.
    polydata = vtkPolyData()
    polydata.SetPoints(points)
    polydata.SetLines(lines)

    # Create the ruled_surface_filter from the polydata containing the lines.
    ruled_surface_filter = vtkRuledSurfaceFilter(input_data=polydata, resolution=(21, 21),
                                                 ruled_mode=RuledSurfaceFilter.RuledMode.VTK_RULED_MODE_RESAMPLE)

    # Create the mapper with the ruled_surface_filter as input.
    mapper = vtkPolyDataMapper()
    ruled_surface_filter >> mapper

    # Create the actor with the mapper.
    actor = vtkActor(mapper=mapper)
    actor.property.color = colors.GetColor3d('Banana')
    actor.property.specular = 0.6
    actor.property.specular_power = 30

    # Add the actor to the display.
    ren.AddActor(actor)

    # Enable the user interface interactor.
    iren.Initialize()
    ren_win.Render()
    # ren.active_camera.position = (3.7, -0.5, -0.5)
    # ren.active_camera.focal_point = (0.5, 0.5, 0.5)
    # ren.active_camera.view_up = (-0.3, 0.1, -1.0)
    ren.active_camera.Azimuth(60)
    ren.active_camera.Elevation(60)
    ren.ResetCamera()

    ren_win.Render()
    iren.Start()


@dataclass(frozen=True)
class RuledSurfaceFilter:
    @dataclass(frozen=True)
    class RuledMode:
        VTK_RULED_MODE_RESAMPLE: int = 0
        VTK_RULED_MODE_POINT_WALK: int = 1


if __name__ == '__main__':
    main()
