#include <vtkDEMReader.h>
#include <vtkImageActor.h>
#include <vtkImageData.h>
#include <vtkImageMapToColors.h>
#include <vtkImageMapper3D.h>
#include <vtkInteractorStyleImage.h>
#include <vtkLookupTable.h>
#include <vtkNamedColors.h>
#include <vtkNew.h>
#include <vtkProperty.h>
#include <vtkRenderWindow.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkRenderer.h>

int main(int argc, char* argv[])
{
  vtkNew<vtkNamedColors> colors;

  // Verify arguments.
  if (argc < 2)
  {
    std::cerr << "Required: filename.dem e.g. SainteHelens.dem" << std::endl;
    return EXIT_FAILURE;
  }

  // Read the file
  vtkNew<vtkDEMReader> reader;
  reader->SetFileName(argv[1]);
  reader->Update();

  vtkNew<vtkLookupTable> lut;
  lut->SetHueRange(0.6, 0);
  lut->SetSaturationRange(1.0, 0);
  lut->SetValueRange(0.5, 1.0);
  lut->SetTableRange(reader->GetOutput()->GetScalarRange());

  // Visualize
  vtkNew<vtkImageMapToColors> mapColors;
  mapColors->SetLookupTable(lut);
  mapColors->SetInputConnection(reader->GetOutputPort());

  // Create an actor.
  vtkNew<vtkImageActor> actor;
  actor->GetMapper()->SetInputConnection(mapColors->GetOutputPort());

  // Setup renderer.
  vtkNew<vtkRenderer> renderer;
  renderer->AddActor(actor);
  renderer->ResetCamera();
  renderer->SetBackground(colors->GetColor3d("SlateGray").GetData());

  // Setup render window.
  vtkNew<vtkRenderWindow> renderWindow;
  renderWindow->AddRenderer(renderer);
  renderWindow->SetWindowName("DEMReader");

  // Setup render window interactor.
  vtkNew<vtkRenderWindowInteractor> renderWindowInteractor;
  vtkNew<vtkInteractorStyleImage> style;

  renderWindowInteractor->SetInteractorStyle(style);

  // Render and start interaction.
  renderWindowInteractor->SetRenderWindow(renderWindow);
  renderWindow->Render();
  renderWindowInteractor->Initialize();

  renderWindowInteractor->Start();

  return EXIT_SUCCESS;
}
