/**
 * @class   vtkCGALPoissonSurfaceReconstructionDelaunay
 * @brief   Surface reconstruction of point sets with oriented normals.
 *
 * vtkCGALPoissonSurfaceReconstructionDelaunay implements a surface reconstruction method
 * which takes as input point sets with oriented normals and computes an implicit function.
 * The input points are assumed to contain no outliers and little noise. The output surface
 * mesh is generated by extracting an isosurface of this function with the CGAL Surface Mesh
 * Generator [4] or potentially with any other surface contouring algorithm.
 * Adapted from
 * https://doc.cgal.org/latest/Poisson_surface_reconstruction_3/Poisson_surface_reconstruction_3_2poisson_reconstruction_function_8cpp-example.html
 */

#ifndef vtkCGALPoissonSurfaceReconstructionDelaunay_h
#define vtkCGALPoissonSurfaceReconstructionDelaunay_h

#include "vtkCGALPolyDataAlgorithm.h"

#include "vtkCGALSRModule.h" // For export macro

class VTKCGALSR_EXPORT vtkCGALPoissonSurfaceReconstructionDelaunay : public vtkCGALPolyDataAlgorithm
{
public:
  static vtkCGALPoissonSurfaceReconstructionDelaunay* New();
  vtkTypeMacro(vtkCGALPoissonSurfaceReconstructionDelaunay, vtkCGALPolyDataAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Get/set the minimum triangle angle in degrees.
   * Default is 20.0.
   **/
  vtkGetMacro(MinTriangleAngle, double);
  vtkSetMacro(MinTriangleAngle, double);

  /**
   * Get/set the maximum triangle size w.r.t. point set average spacing.
   * Default is 2.0.
   **/
  vtkGetMacro(MaxTriangleSize, double);
  vtkSetMacro(MaxTriangleSize, double);

  /**
   * Get/set the surface approximation error w.r.t. point set average spacing.
   * Default is 0.375.
   **/
  vtkGetMacro(Distance, double);
  vtkSetMacro(Distance, double);

  /**
   * Get/set (re-)generating surface normals.
   * Default is true.
   **/
  vtkGetMacro(GenerateSurfaceNormals, bool);
  vtkSetMacro(GenerateSurfaceNormals, bool);

protected:
  vtkCGALPoissonSurfaceReconstructionDelaunay();
  ~vtkCGALPoissonSurfaceReconstructionDelaunay() override = default;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

private:
  vtkCGALPoissonSurfaceReconstructionDelaunay(const vtkCGALPoissonSurfaceReconstructionDelaunay&) = delete;
  void operator=(const vtkCGALPoissonSurfaceReconstructionDelaunay&)       = delete;

  double MinTriangleAngle;
  double MaxTriangleSize;
  double Distance;
  bool GenerateSurfaceNormals;
};

#endif
