// -*- c++ -*-
/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCGALXYZWriter.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

/*-------------------------------------------------------------------------
  Copyright 2008 Sandia Corporation.
  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
  the U.S. Government retains certain rights in this software.
-------------------------------------------------------------------------*/

#include "vtkCGALXYZWriter.h"

#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkObjectFactory.h"
#include "vtkPolyData.h"
#include <vtkLogger.h>

#include <vtkPointData.h>
#include <vtksys/SystemTools.hxx>

#include <CGAL/IO/write_xyz_points.h>

typedef CGAL_Kernel::Point_3                                 Point;
typedef CGAL_Kernel::Vector_3                                Vector;
typedef std::pair<Point, Vector>                             Point_with_normal;
typedef CGAL::First_of_pair_property_map<Point_with_normal>  Point_map;
typedef CGAL::Second_of_pair_property_map<Point_with_normal> Normal_map;
typedef std::vector<Point_with_normal>                       PointList;
typedef std::pair<Point, Vector>                             Pwn;

//=============================================================================
vtkStandardNewMacro(vtkCGALXYZWriter);

//------------------------------------------------------------------------------
vtkCGALXYZWriter::vtkCGALXYZWriter()
{
  this->SetNumberOfInputPorts(1);
  this->SetNumberOfOutputPorts(0);

  this->FileName = nullptr;
}

//------------------------------------------------------------------------------
vtkCGALXYZWriter::~vtkCGALXYZWriter()
{
  this->SetFileName(nullptr);
}

//------------------------------------------------------------------------------
void vtkCGALXYZWriter::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "FileName: " << (this->FileName ? this->FileName : "(nullptr)") << endl;
}

//------------------------------------------------------------------------------
int vtkCGALXYZWriter::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* vtkNotUsed(outputVector))
{
  // Get the input and output data objects.
  vtkPolyData* input = vtkPolyData::GetData(inputVector[0]);

  if (!input)
  {
    vtkErrorMacro(<< "Bad input type.");
    return 0;
  }

  auto normals = input->GetPointData()->GetNormals();
  if (!normals)
  {
    vtkErrorMacro("Point normals required.");
    return 0;
  }

  // Create the surface mesh for CGAL
  // ----------------------------------

  std::unique_ptr<Vespa_surface> cgalMesh = std::make_unique<Vespa_surface>();
  this->toCGAL(input, cgalMesh.get());

  try
  {

    std::vector<Pwn> points;

    for (int k = 0; k < input->GetNumberOfPoints(); k++)
    {
      auto pin = input->GetPoint(k);
      auto vin = normals->GetTuple3(k);
      points.emplace_back(Point(pin[0], pin[1], pin[2]), Vector(vin[0], vin[1], vin[2]));
    }

    if (!CGAL::IO::write_XYZ(this->FileName, points,
          CGAL::parameters::point_map(CGAL::First_of_pair_property_map<Pwn>())
            .normal_map(CGAL::Second_of_pair_property_map<Pwn>())
            .stream_precision(17)))
    {
      vtkErrorMacro("Cannot write file.");
      return 0;
    }
  }
  catch (std::exception& e)
  {
    vtkErrorMacro("CGAL Exception: " << e.what());
    return 0;
  }

  return 1;
}


//------------------------------------------------------------------------------
int vtkCGALXYZWriter::Write()
{
  // Make sure we have input.
  if (this->GetNumberOfInputConnections(0) < 1)
  {
    vtkErrorMacro("No input provided!");
    return 0;
  }

  // always write even if the data hasn't changed
  this->Modified();
  this->Update();
  return 1;
}
