/**
 * @struct vtkMomentInvariantData
 * @brief a data structure that stores pre-computed moment invariants
 * This struct stores the necessary information (order, productInfo and contractionInfo) for constructing a 
 * polynomial. Invariants are pre-computed and encoded as a string. 
 * This struct provides funtions to decode both the generator invariants and the Lanbein invariants.
 */

#ifndef VTKMOMENTINVARIANTDATA_H
#define VTKMOMENTINVARIANTDATA_H
#ifndef __VTK_WRAP__

#include "vtkMomentsPolynomial.h"
struct vtkMomentInvariantData {
  /**
   * Order of invariants
   */    
  unsigned order;

  /**
   * encoded homogeneous invariants 
   */
  string homogeneousString;

  /**
   * encoded mixed invariants or Lanbein invariant
   */
  string mixedString;

  /**
   * This function decodes homogeneous invariants up to order o.
   */
  list<list<vtkMomentsPolynomial>> getHomogeneousInvariants(unsigned o) {
    list<list<vtkMomentsPolynomial>> invariants;
    stringstream ss;
    ss << homogeneousString;
    ss >> invariants;
    
    if (o < order)
      for (auto it1 = invariants.begin(); it1 != invariants.end();) {
	for (auto it2 = (*it1).begin(); it2 != (*it1).end();)
	  if ((*it2).getMaximumTensorOrder() > o)
	    it2 = (*it1).erase(it2);
	  else
	    ++it2;
	
	if ((*it1).empty())
	  it1 = invariants.erase(it1);
	else
	  ++it1;
      }
	        
    return invariants;
  }

  /**
   * This function decodes mixed invariants up to order o.
   */
  list<list<list<vtkMomentsPolynomial>>> getMixedInvariants(unsigned o) {
    list<list<list<vtkMomentsPolynomial>>> invariants;
    stringstream ss;
    ss << mixedString;
    ss >> invariants;

    if (o < order)
      for (auto it1 = invariants.begin(); it1 != invariants.end();) {	
	for (auto it2 = (*it1).begin(); it2 != (*it1).end();) {
	  for (auto it3 = (*it2).begin(); it3 != (*it2).end();)
	    if ((*it3).getMaximumTensorOrder() > o)
	      it3 = (*it2).erase(it3);
	    else
	      ++it3;
	
	  if ((*it2).empty())
	    it2 = (*it1).erase(it2);
	  else
	    ++it2;
	}
	
	if ((*it1).empty())
	    it1 = invariants.erase(it1);
	  else
	    ++it1;
      }
    
    return invariants;
  }
  
  /**
   * This function decodes Langbein invariants up to order o.
   */  
  list<vtkMomentsPolynomial> getLangbeinInvariants(unsigned o) {
    if (mixedString.empty()) {
      cerr << "getLangbeinInvariants: mixedString is empty\n";
      exit(0);
    }
      
    list<vtkMomentsPolynomial> invariants;
    stringstream ss;
    ss << mixedString;
    ss >> invariants;

    if (o < order)
      for (auto it = invariants.cbegin(); it != invariants.cend();) {
	if ((*it).getMaximumTensorOrder() > o)
	  it = invariants.erase(it);
	else
	  ++it;
      }
    
    return invariants;
  }
};

/**
 * encoded invariants
 */
static vtkMomentInvariantData generator2DScalarOrder6 = {6, "7 1 2 0 1 0  0    1 2 0 2 1 1  2 0 1    2 2 0 1 2  2 0 1   2 0 2 2 2  4 0 2 0 1    3 2 0 2 3 3  6 0 1 0 1 0 1   2 0 2 3 3  6 0 3 0 2 0 1   2 0 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1    4 2 0 1 4  4 0 1 0 1   2 0 2 4 4  8 0 1 0 2 0 1 0 1   2 0 2 4 4  8 0 4 0 3 0 2 0 1   2 0 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1    5 2 0 2 5 5  10 0 1 0 1 0 1 0 1 0 1   2 0 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 0 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 0 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 1 0 1 0 1 0 1   2 0 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 3 0 2 0 1 0 1    6 2 0 1 6  6 0 1 0 1 0 1   2 0 2 6 6  12 0 1 0 1 0 2 0 1 0 1 0 1   2 0 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 0 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 0 3 6 6 6  18 0 1 0 1 0 2 0 1 0 1 0 2 0 1 0 1 0 1   2 0 3 6 6 6  18 0 1 0 1 0 2 0 6 0 5 0 4 0 3 0 2 0 1", "5 1 1 2 0 3 1 1 2  4 0 2 0 1     2 1 2 0 2 1 3  4 0 1 0 1    1 2 0 3 2 3 3  8 0 2 0 1 0 1 0 1     3 1 2 0 3 1 1 4  6 0 2 0 1 0 1    1 2 0 2 2 4  6 0 2 0 1 0 1    1 2 0 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1     4 1 2 0 2 1 5  6 0 1 0 1 0 1    1 2 0 3 2 5 5  12 0 2 0 1 0 1 0 1 0 1 0 1    1 2 0 2 3 5  8 0 3 0 2 0 1 0 1    1 2 0 3 4 5 5  14 0 4 0 3 0 2 0 1 0 1 0 1 0 1     5 1 2 0 3 1 1 6  8 0 2 0 1 0 1 0 1    1 2 0 2 2 6  8 0 2 0 1 0 1 0 1    1 2 0 3 3 3 6  12 0 6 0 5 0 4 0 3 0 2 0 1    1 2 0 2 4 6  10 0 4 0 3 0 2 0 1 0 1    1 2 0 3 5 5 6  16 0 1 0 1 0 6 0 5 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData generator2DVectorOrder5 = {5, "6 1 2 1 2 1 1  2 0 1    3 2 1 1 2  2 0 1   2 1 2 2 2  4 0 2 0 1   2 1 2 2 2  4 0 3 0 1    5 2 1 2 3 3  6 0 1 0 1 0 1   2 1 2 3 3  6 0 1 0 3 0 1   2 1 2 3 3  6 0 2 0 1 0 1   2 1 2 3 3  6 0 3 0 2 0 1   2 1 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1    7 2 1 1 4  4 0 1 0 1   2 1 2 4 4  8 0 1 0 2 0 1 0 1   2 1 2 4 4  8 0 1 0 2 0 3 0 1   2 1 2 4 4  8 0 1 0 5 0 1 0 1   2 1 2 4 4  8 0 3 0 2 0 1 0 1   2 1 2 4 4  8 0 4 0 3 0 2 0 1   2 1 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1    9 2 1 2 5 5  10 0 1 0 1 0 1 0 1 0 1   2 1 2 5 5  10 0 1 0 1 0 5 0 1 0 1   2 1 2 5 5  10 0 1 0 2 0 1 0 1 0 1   2 1 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 1 2 5 5  10 0 1 0 3 0 2 0 3 0 1   2 1 2 5 5  10 0 4 0 3 0 2 0 1 0 1   2 1 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 1 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 1 0 1 0 1 0 1   2 1 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 3 0 2 0 1 0 1    11 2 1 1 6  6 0 1 0 1 0 1   2 1 2 6 6  12 0 1 0 1 0 2 0 1 0 1 0 1   2 1 2 6 6  12 0 1 0 1 0 2 0 5 0 1 0 1   2 1 2 6 6  12 0 1 0 1 0 7 0 1 0 1 0 1   2 1 2 6 6  12 0 1 0 3 0 2 0 1 0 1 0 1   2 1 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 1 2 6 6  12 0 1 0 4 0 3 0 2 0 3 0 1   2 1 2 6 6  12 0 5 0 4 0 3 0 2 0 1 0 1   2 1 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 1 3 6 6 6  18 0 1 0 1 0 2 0 1 0 1 0 2 0 1 0 1 0 1   2 1 3 6 6 6  18 0 1 0 1 0 2 0 1 0 4 0 3 0 2 0 1 0 1", "5 1 1 2 1 3 1 1 2  4 0 2 0 1     2 1 2 1 2 1 3  4 0 1 0 1    1 2 1 3 2 3 3  8 0 2 0 1 0 1 0 1     3 1 2 1 3 1 1 4  6 0 2 0 1 0 1    1 2 1 2 2 4  6 0 2 0 1 0 1    1 2 1 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1     4 1 2 1 2 1 5  6 0 1 0 1 0 1    1 2 1 3 2 5 5  12 0 2 0 1 0 1 0 1 0 1 0 1    1 2 1 2 3 5  8 0 3 0 2 0 1 0 1    1 2 1 3 4 5 5  14 0 4 0 3 0 2 0 1 0 1 0 1 0 1     5 1 2 1 3 1 1 6  8 0 2 0 1 0 1 0 1    1 2 1 2 2 6  8 0 2 0 1 0 1 0 1    1 2 1 3 3 3 6  12 0 6 0 5 0 4 0 3 0 2 0 1    1 2 1 2 4 6  10 0 4 0 3 0 2 0 1 0 1    1 2 1 3 5 5 6  16 0 1 0 1 0 6 0 5 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData generator2DMatrixOrder4 = {4, "5 3 2 2 1 2  2 0 1   2 2 2 2 2  4 0 2 0 1   2 2 2 2 2  4 0 3 0 1    7 2 2 2 3 3  6 0 1 0 1 0 1   2 2 2 3 3  6 0 1 0 2 0 1   2 2 2 3 3  6 0 1 0 3 0 1   2 2 2 3 3  6 0 2 0 1 0 1   2 2 2 3 3  6 0 2 0 2 0 1   2 2 2 3 3  6 0 3 0 2 0 1   2 2 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1    11 2 2 1 4  4 0 1 0 1   2 2 1 4  4 0 2 0 1   2 2 2 4 4  8 0 1 0 2 0 1 0 1   2 2 2 4 4  8 0 1 0 2 0 2 0 1   2 2 2 4 4  8 0 1 0 2 0 3 0 1   2 2 2 4 4  8 0 1 0 4 0 1 0 1   2 2 2 4 4  8 0 1 0 5 0 1 0 1   2 2 2 4 4  8 0 2 0 2 0 1 0 1   2 2 2 4 4  8 0 2 0 2 0 2 0 1   2 2 2 4 4  8 0 4 0 3 0 2 0 1   2 2 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1    15 2 2 2 5 5  10 0 1 0 1 0 1 0 1 0 1   2 2 2 5 5  10 0 1 0 1 0 1 0 2 0 1   2 2 2 5 5  10 0 1 0 1 0 4 0 1 0 1   2 2 2 5 5  10 0 1 0 1 0 5 0 1 0 1   2 2 2 5 5  10 0 1 0 2 0 1 0 1 0 1   2 2 2 5 5  10 0 1 0 2 0 1 0 2 0 1   2 2 2 5 5  10 0 1 0 2 0 4 0 1 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 2 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 3 0 1   2 2 2 5 5  10 0 3 0 3 0 2 0 1 0 1   2 2 2 5 5  10 0 3 0 3 0 2 0 2 0 1   2 2 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 2 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 1 0 1 0 1 0 1   2 2 4 5 5 5 5  20 0 1 0 1 0 1 0 1 0 2 0 1 0 3 0 2 0 1 0 1    19 2 2 1 6  6 0 1 0 1 0 1   2 2 1 6  6 0 1 0 2 0 1   2 2 2 6 6  12 0 1 0 1 0 2 0 1 0 1 0 1   2 2 2 6 6  12 0 1 0 1 0 2 0 1 0 2 0 1   2 2 2 6 6  12 0 1 0 1 0 2 0 4 0 1 0 1   2 2 2 6 6  12 0 1 0 1 0 2 0 5 0 1 0 1   2 2 2 6 6  12 0 1 0 1 0 6 0 1 0 1 0 1   2 2 2 6 6  12 0 1 0 1 0 7 0 1 0 1 0 1   2 2 2 6 6  12 0 1 0 2 0 2 0 1 0 1 0 1   2 2 2 6 6  12 0 1 0 2 0 2 0 1 0 2 0 1   2 2 2 6 6  12 0 1 0 2 0 2 0 4 0 1 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 2 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 3 0 1   2 2 2 6 6  12 0 4 0 4 0 3 0 2 0 1 0 1   2 2 2 6 6  12 0 4 0 4 0 3 0 2 0 2 0 1   2 2 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 2 3 6 6 6  18 0 1 0 1 0 2 0 1 0 1 0 2 0 1 0 1 0 1   2 2 3 6 6 6  18 0 1 0 1 0 2 0 1 0 4 0 3 0 2 0 1 0 1", "4 1 1 2 2 3 2 3 3  8 0 2 0 1 0 1 0 1     2 1 2 2 2 2 4  6 0 2 0 1 0 1    1 2 2 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1     3 1 2 2 3 2 5 5  12 0 2 0 1 0 1 0 1 0 1 0 1    1 2 2 2 3 5  8 0 3 0 2 0 1 0 1    1 2 2 3 4 5 5  14 0 4 0 3 0 2 0 1 0 1 0 1 0 1     4 1 2 2 2 2 6  8 0 2 0 1 0 1 0 1    1 2 2 3 3 3 6  12 0 6 0 5 0 4 0 3 0 2 0 1    1 2 2 2 4 6  10 0 4 0 3 0 2 0 1 0 1    1 2 2 3 5 5 6  16 0 1 0 1 0 6 0 5 0 4 0 3 0 2 0 1"};
static vtkMomentInvariantData generator3DScalarOrder4 = {4, "5 1 3 0 1 0  0    1 3 0 2 1 1  2 0 1    3 3 0 1 2  2 0 1   3 0 2 2 2  4 0 2 0 1   3 0 3 2 2 2  6 0 2 0 2 0 1    7 3 0 2 3 3  6 0 1 0 1 0 1   3 0 2 3 3  6 0 3 0 2 0 1   3 0 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1   3 0 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 1 0 1   3 0 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 2 0 1   3 0 4 3 3 3 3  12 0 3 0 2 0 2 0 3 0 2 0 1   3 0 6 3 3 3 3 3 3  18 0 1 0 1 0 2 0 1 0 1 0 2 0 1 0 1 0 1    12 3 0 1 4  4 0 1 0 1   3 0 2 4 4  8 0 1 0 2 0 1 0 1   3 0 2 4 4  8 0 4 0 3 0 2 0 1   3 0 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1   3 0 3 4 4 4  12 0 1 0 2 0 4 0 1 0 1 0 1   3 0 3 4 4 4  12 0 1 0 2 0 4 0 3 0 2 0 1   3 0 3 4 4 4  12 0 4 0 3 0 4 0 3 0 2 0 1   3 0 4 4 4 4 4  16 0 1 0 2 0 1 0 2 0 1 0 2 0 1 0 1   3 0 4 4 4 4 4  16 0 1 0 2 0 1 0 2 0 4 0 1 0 1 0 1   3 0 4 4 4 4 4  16 0 1 0 2 0 1 0 2 0 4 0 3 0 2 0 1   3 0 4 4 4 4 4  16 0 1 0 2 0 4 0 1 0 4 0 3 0 2 0 1   3 0 4 4 4 4 4  16 0 4 0 3 0 2 0 2 0 4 0 3 0 2 0 1", "3 1 2 3 0 3 1 1 2  4 0 2 0 1   3 0 4 1 1 2 2  6 0 2 0 2 0 1     2 2 3 0 4 1 1 3 3  8 0 2 0 1 0 1 0 1   3 0 2 1 3  4 0 1 0 1    3 3 0 3 2 3 3  8 0 2 0 1 0 1 0 1   3 0 3 2 3 3  8 0 2 0 3 0 1 0 1   3 0 3 2 3 3  8 0 2 0 3 0 2 0 1     3 2 3 0 3 1 1 4  6 0 2 0 1 0 1   3 0 4 1 1 4 4  10 0 2 0 1 0 2 0 1 0 1    3 3 0 2 2 4  6 0 2 0 1 0 1   3 0 3 2 2 4  8 0 2 0 2 0 1 0 1   3 0 3 2 2 4  8 0 4 0 3 0 2 0 1    3 3 0 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1   3 0 3 3 3 4  10 0 1 0 1 0 2 0 1 0 1   3 0 3 3 3 4  10 0 1 0 4 0 1 0 1 0 1"};

static vtkMomentInvariantData generator3DVectorOrder3 = {3, "4 1 3 1 2 1 1  2 0 1    6 3 1 1 2  2 0 1   3 1 2 2 2  4 0 2 0 1   3 1 2 2 2  4 0 3 0 1   3 1 3 2 2 2  6 0 2 0 2 0 1   3 1 3 2 2 2  6 0 3 0 2 0 1   3 1 4 2 2 2 2  8 0 2 0 2 0 2 0 1    15 3 1 2 3 3  6 0 1 0 1 0 1   3 1 2 3 3  6 0 1 0 3 0 1   3 1 2 3 3  6 0 2 0 1 0 1   3 1 2 3 3  6 0 3 0 2 0 1   3 1 2 3 3  6 0 3 0 3 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 3 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 2 0 1 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 2 0 3 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 1 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 2 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 3 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 5 0 1 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 2 0 5 0 2 0 1   3 1 4 3 3 3 3  12 0 1 0 1 0 4 0 1 0 1 0 1    27 3 1 1 4  4 0 1 0 1   3 1 2 4 4  8 0 1 0 2 0 1 0 1   3 1 2 4 4  8 0 1 0 2 0 3 0 1   3 1 2 4 4  8 0 1 0 5 0 1 0 1   3 1 2 4 4  8 0 3 0 2 0 1 0 1   3 1 2 4 4  8 0 4 0 3 0 2 0 1   3 1 2 4 4  8 0 4 0 3 0 3 0 1   3 1 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 1 0 2 0 3 0 1   3 1 3 4 4 4  12 0 1 0 2 0 1 0 5 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 3 0 2 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 1 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 1 0 3 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 2 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 2 0 3 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 3 0 2 0 1   3 1 3 4 4 4  12 0 1 0 2 0 4 0 3 0 3 0 1   3 1 3 4 4 4  12 0 1 0 2 0 7 0 2 0 1 0 1   3 1 3 4 4 4  12 0 1 0 2 0 7 0 3 0 2 0 1   3 1 3 4 4 4  12 0 1 0 5 0 1 0 2 0 1 0 1   3 1 3 4 4 4  12 0 1 0 5 0 4 0 1 0 3 0 1   3 1 3 4 4 4  12 0 1 0 5 0 4 0 3 0 2 0 1   3 1 3 4 4 4  12 0 3 0 2 0 1 0 2 0 1 0 1   3 1 3 4 4 4  12 0 3 0 2 0 4 0 3 0 2 0 1   3 1 3 4 4 4  12 0 3 0 2 0 4 0 3 0 3 0 1   3 1 3 4 4 4  12 0 3 0 2 0 7 0 3 0 2 0 1   3 1 3 4 4 4  12 0 4 0 3 0 4 0 3 0 2 0 1", "3 1 2 3 1 3 1 1 2  4 0 2 0 1   3 1 4 1 1 2 2  6 0 2 0 2 0 1     2 2 3 1 4 1 1 3 3  8 0 2 0 1 0 1 0 1   3 1 2 1 3  4 0 1 0 1    3 3 1 3 2 3 3  8 0 2 0 1 0 1 0 1   3 1 3 2 3 3  8 0 2 0 1 0 3 0 1   3 1 3 2 3 3  8 0 2 0 2 0 1 0 1     3 2 3 1 3 1 1 4  6 0 2 0 1 0 1   3 1 3 1 1 4  6 0 2 0 3 0 1    3 3 1 2 2 4  6 0 2 0 1 0 1   3 1 2 2 4  6 0 2 0 3 0 1   3 1 2 2 4  6 0 5 0 1 0 1    3 3 1 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1   3 1 3 3 3 4  10 0 1 0 1 0 2 0 1 0 1   3 1 3 3 3 4  10 0 1 0 1 0 2 0 3 0 1"};

static vtkMomentInvariantData generator3DMatrixOrder2 = {2, "3 6 3 2 1 2  2 0 1   3 2 2 2 2  4 0 2 0 1   3 2 2 2 2  4 0 3 0 1   3 2 3 2 2 2  6 0 2 0 2 0 1   3 2 3 2 2 2  6 0 3 0 2 0 1   3 2 4 2 2 2 2  8 0 2 0 2 0 2 0 1    24 3 2 2 3 3  6 0 1 0 1 0 1   3 2 2 3 3  6 0 1 0 2 0 1   3 2 2 3 3  6 0 1 0 3 0 1   3 2 2 3 3  6 0 2 0 1 0 1   3 2 2 3 3  6 0 2 0 2 0 1   3 2 2 3 3  6 0 3 0 1 0 1   3 2 2 3 3  6 0 3 0 2 0 1   3 2 2 3 3  6 0 3 0 3 0 1   3 2 2 3 3  6 0 4 0 2 0 1   3 2 2 3 3  6 0 4 0 3 0 1   3 2 2 3 3  6 0 5 0 3 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 1 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 2 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 1 0 3 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 2 0 1 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 2 0 2 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 2 0 3 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 1 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 2 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 3 0 3 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 4 0 1 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 4 0 2 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 4 0 3 0 1   3 2 4 3 3 3 3  12 0 1 0 1 0 2 0 5 0 1 0 1    51 3 2 1 4  4 0 1 0 1   3 2 1 4  4 0 2 0 1   3 2 2 4 4  8 0 1 0 2 0 1 0 1   3 2 2 4 4  8 0 1 0 2 0 2 0 1   3 2 2 4 4  8 0 1 0 2 0 3 0 1   3 2 2 4 4  8 0 1 0 4 0 1 0 1   3 2 2 4 4  8 0 1 0 4 0 3 0 1   3 2 2 4 4  8 0 1 0 5 0 1 0 1   3 2 2 4 4  8 0 1 0 5 0 3 0 1   3 2 2 4 4  8 0 2 0 2 0 1 0 1   3 2 2 4 4  8 0 2 0 2 0 2 0 1   3 2 2 4 4  8 0 2 0 2 0 3 0 1   3 2 2 4 4  8 0 2 0 4 0 1 0 1   3 2 2 4 4  8 0 2 0 5 0 1 0 1   3 2 2 4 4  8 0 3 0 2 0 1 0 1   3 2 2 4 4  8 0 3 0 2 0 2 0 1   3 2 2 4 4  8 0 3 0 4 0 1 0 1   3 2 2 4 4  8 0 4 0 3 0 1 0 1   3 2 2 4 4  8 0 4 0 3 0 2 0 1   3 2 2 4 4  8 0 4 0 3 0 3 0 1   3 2 2 4 4  8 0 4 0 4 0 2 0 1   3 2 2 4 4  8 0 4 0 4 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 2 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 2 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 4 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 4 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 5 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 1 0 5 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 2 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 2 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 2 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 4 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 4 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 5 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 2 0 5 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 2 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 2 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 2 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 4 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 5 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 3 0 5 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 1 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 1 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 1 0 3 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 2 0 1 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 2 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 4 0 3 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 6 0 3 0 2 0 1   3 2 3 4 4 4  12 0 1 0 2 0 7 0 3 0 2 0 1   3 2 3 4 4 4  12 0 1 0 4 0 4 0 3 0 2 0 1", "2 1 3 3 2 3 2 3 3  8 0 2 0 1 0 1 0 1   3 2 3 2 3 3  8 0 2 0 1 0 2 0 1   3 2 3 2 3 3  8 0 2 0 1 0 3 0 1     2 3 3 2 2 2 4  6 0 2 0 1 0 1   3 2 2 2 4  6 0 2 0 2 0 1   3 2 2 2 4  6 0 2 0 3 0 1    3 3 2 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1   3 2 3 3 3 4  10 0 1 0 1 0 2 0 1 0 1   3 2 3 3 3 4  10 0 1 0 1 0 2 0 2 0 1"};


static vtkMomentInvariantData langbein2DScalarOrder6 = {6, "", "27 2 0 1 0  0   2 0 2 1 1  2 0 1   2 0 1 2  2 0 1   2 0 2 2 2  4 0 2 0 1   2 0 3 1 1 2  4 0 2 0 1   2 0 2 1 3  4 0 1 0 1   2 0 2 3 3  6 0 1 0 1 0 1   2 0 2 3 3  6 0 3 0 2 0 1   2 0 3 1 2 3  6 0 3 0 2 0 1   2 0 1 4  4 0 1 0 1   2 0 2 2 4  6 0 2 0 1 0 1   2 0 2 4 4  8 0 1 0 2 0 1 0 1   2 0 2 4 4  8 0 4 0 3 0 2 0 1   2 0 3 1 3 4  8 0 4 0 3 0 2 0 1   2 0 2 1 5  6 0 1 0 1 0 1   2 0 2 3 5  8 0 1 0 1 0 1 0 1   2 0 2 3 5  8 0 3 0 2 0 1 0 1   2 0 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 0 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 0 3 1 4 5  10 0 5 0 4 0 3 0 2 0 1   2 0 1 6  6 0 1 0 1 0 1   2 0 2 2 6  8 0 2 0 1 0 1 0 1   2 0 2 4 6  10 0 1 0 2 0 1 0 1 0 1   2 0 2 4 6  10 0 4 0 3 0 2 0 1 0 1   2 0 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 0 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 0 3 1 5 6  12 0 6 0 5 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData langbein2DVectorOrder5 = {5, "", "41 2 1 2 1 1  2 0 1   2 1 1 2  2 0 1   2 1 2 2 2  4 0 2 0 1   2 1 2 2 2  4 0 3 0 1   2 1 3 1 1 2  4 0 2 0 1   2 1 2 1 3  4 0 1 0 1   2 1 2 1 3  4 0 3 0 1   2 1 2 3 3  6 0 1 0 1 0 1   2 1 2 3 3  6 0 1 0 3 0 1   2 1 2 3 3  6 0 3 0 2 0 1   2 1 3 1 2 3  6 0 3 0 2 0 1   2 1 1 4  4 0 1 0 1   2 1 2 2 4  6 0 2 0 1 0 1   2 1 2 2 4  6 0 2 0 3 0 1   2 1 2 2 4  6 0 5 0 1 0 1   2 1 2 4 4  8 0 1 0 2 0 1 0 1   2 1 2 4 4  8 0 1 0 2 0 3 0 1   2 1 2 4 4  8 0 4 0 3 0 2 0 1   2 1 3 1 3 4  8 0 4 0 3 0 2 0 1   2 1 2 1 5  6 0 1 0 1 0 1   2 1 2 1 5  6 0 5 0 1 0 1   2 1 2 3 5  8 0 1 0 1 0 1 0 1   2 1 2 3 5  8 0 1 0 5 0 1 0 1   2 1 2 3 5  8 0 3 0 2 0 1 0 1   2 1 2 3 5  8 0 3 0 2 0 3 0 1   2 1 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 1 2 5 5  10 0 1 0 3 0 2 0 3 0 1   2 1 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 1 3 1 4 5  10 0 5 0 4 0 3 0 2 0 1   2 1 1 6  6 0 1 0 1 0 1   2 1 2 2 6  8 0 2 0 1 0 1 0 1   2 1 2 2 6  8 0 2 0 5 0 1 0 1   2 1 2 2 6  8 0 7 0 1 0 1 0 1   2 1 2 4 6  10 0 1 0 2 0 1 0 1 0 1   2 1 2 4 6  10 0 1 0 2 0 5 0 1 0 1   2 1 2 4 6  10 0 4 0 3 0 2 0 1 0 1   2 1 2 4 6  10 0 4 0 3 0 2 0 3 0 1   2 1 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 1 2 6 6  12 0 1 0 4 0 3 0 2 0 3 0 1   2 1 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 1 3 1 5 6  12 0 6 0 5 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData langbein2DMatrixOrder4 = {4, "", "59 2 2 1 2  2 0 1   2 2 2 2 2  4 0 2 0 1   2 2 2 2 2  4 0 3 0 1   2 2 2 3 3  6 0 1 0 1 0 1   2 2 2 3 3  6 0 1 0 2 0 1   2 2 2 3 3  6 0 1 0 3 0 1   2 2 2 3 3  6 0 2 0 1 0 1   2 2 2 3 3  6 0 2 0 2 0 1   2 2 2 3 3  6 0 3 0 2 0 1   2 2 3 2 3 3  8 0 2 0 1 0 1 0 1   2 2 3 2 3 3  8 0 2 0 1 0 2 0 1   2 2 1 4  4 0 1 0 1   2 2 1 4  4 0 2 0 1   2 2 2 2 4  6 0 2 0 1 0 1   2 2 2 2 4  6 0 2 0 2 0 1   2 2 2 2 4  6 0 2 0 3 0 1   2 2 2 2 4  6 0 4 0 1 0 1   2 2 2 2 4  6 0 5 0 1 0 1   2 2 2 4 4  8 0 1 0 2 0 1 0 1   2 2 2 4 4  8 0 1 0 2 0 2 0 1   2 2 2 4 4  8 0 1 0 2 0 3 0 1   2 2 2 4 4  8 0 4 0 3 0 2 0 1   2 2 3 2 2 4  8 0 4 0 3 0 2 0 1   2 2 2 3 5  8 0 1 0 1 0 1 0 1   2 2 2 3 5  8 0 1 0 1 0 2 0 1   2 2 2 3 5  8 0 1 0 4 0 1 0 1   2 2 2 3 5  8 0 1 0 5 0 1 0 1   2 2 2 3 5  8 0 2 0 1 0 1 0 1   2 2 2 3 5  8 0 2 0 1 0 2 0 1   2 2 2 3 5  8 0 2 0 4 0 1 0 1   2 2 2 3 5  8 0 2 0 5 0 1 0 1   2 2 2 3 5  8 0 3 0 2 0 1 0 1   2 2 2 3 5  8 0 3 0 2 0 2 0 1   2 2 2 3 5  8 0 3 0 2 0 3 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 1 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 2 0 1   2 2 2 5 5  10 0 1 0 3 0 2 0 3 0 1   2 2 2 5 5  10 0 5 0 4 0 3 0 2 0 1   2 2 3 2 3 5  10 0 5 0 4 0 3 0 2 0 1   2 2 1 6  6 0 1 0 1 0 1   2 2 1 6  6 0 1 0 2 0 1   2 2 2 2 6  8 0 2 0 1 0 1 0 1   2 2 2 2 6  8 0 2 0 1 0 2 0 1   2 2 2 2 6  8 0 2 0 4 0 1 0 1   2 2 2 2 6  8 0 2 0 5 0 1 0 1   2 2 2 2 6  8 0 6 0 1 0 1 0 1   2 2 2 2 6  8 0 7 0 1 0 1 0 1   2 2 2 4 6  10 0 1 0 2 0 1 0 1 0 1   2 2 2 4 6  10 0 1 0 2 0 1 0 2 0 1   2 2 2 4 6  10 0 1 0 2 0 4 0 1 0 1   2 2 2 4 6  10 0 1 0 2 0 5 0 1 0 1   2 2 2 4 6  10 0 4 0 3 0 2 0 1 0 1   2 2 2 4 6  10 0 4 0 3 0 2 0 2 0 1   2 2 2 4 6  10 0 4 0 3 0 2 0 3 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 1 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 2 0 1   2 2 2 6 6  12 0 1 0 4 0 3 0 2 0 3 0 1   2 2 2 6 6  12 0 6 0 5 0 4 0 3 0 2 0 1   2 2 3 2 4 6  12 0 6 0 5 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData langbein3DScalarOrder4 = {4, "", "32 3 0 1 0  0   3 0 2 1 1  2 0 1   3 0 1 2  2 0 1   3 0 2 2 2  4 0 2 0 1   3 0 3 1 1 2  4 0 2 0 1   3 0 3 2 2 2  6 0 2 0 2 0 1   3 0 4 1 1 2 2  6 0 2 0 2 0 1   3 0 2 1 3  4 0 1 0 1   3 0 2 3 3  6 0 1 0 1 0 1   3 0 2 3 3  6 0 3 0 2 0 1   3 0 3 1 2 3  6 0 1 0 1 0 1   3 0 3 1 2 3  6 0 3 0 2 0 1   3 0 3 2 3 3  8 0 2 0 1 0 1 0 1   3 0 3 2 3 3  8 0 2 0 3 0 2 0 1   3 0 4 1 1 1 3  6 0 3 0 2 0 1   3 0 4 1 1 3 3  8 0 2 0 1 0 1 0 1   3 0 4 1 1 3 3  8 0 2 0 3 0 2 0 1   3 0 1 4  4 0 1 0 1   3 0 2 2 4  6 0 2 0 1 0 1   3 0 2 4 4  8 0 1 0 2 0 1 0 1   3 0 2 4 4  8 0 4 0 3 0 2 0 1   3 0 3 1 1 4  6 0 2 0 1 0 1   3 0 3 1 3 4  8 0 1 0 2 0 1 0 1   3 0 3 1 3 4  8 0 4 0 1 0 1 0 1   3 0 3 1 3 4  8 0 4 0 3 0 2 0 1   3 0 3 2 2 4  8 0 4 0 3 0 2 0 1   3 0 3 2 4 4  10 0 2 0 1 0 2 0 1 0 1   3 0 3 2 4 4  10 0 2 0 4 0 3 0 2 0 1   3 0 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1   3 0 3 3 3 4  10 0 3 0 4 0 3 0 2 0 1   3 0 3 4 4 4  12 0 1 0 2 0 1 0 2 0 1 0 1   3 0 3 4 4 4  12 0 1 0 2 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData langbein3DVectorOrder3 = {3, "", "57 3 1 2 1 1  2 0 1   3 1 1 2  2 0 1   3 1 2 2 2  4 0 2 0 1   3 1 2 2 2  4 0 3 0 1   3 1 3 1 1 2  4 0 2 0 1   3 1 3 2 2 2  6 0 2 0 2 0 1   3 1 3 2 2 2  6 0 3 0 2 0 1   3 1 4 1 1 2 2  6 0 2 0 2 0 1   3 1 4 1 1 2 2  6 0 2 0 3 0 1   3 1 2 1 3  4 0 1 0 1   3 1 2 1 3  4 0 3 0 1   3 1 2 3 3  6 0 1 0 1 0 1   3 1 2 3 3  6 0 1 0 3 0 1   3 1 2 3 3  6 0 2 0 1 0 1   3 1 2 3 3  6 0 3 0 2 0 1   3 1 2 3 3  6 0 3 0 3 0 1   3 1 3 1 2 3  6 0 1 0 1 0 1   3 1 3 1 2 3  6 0 3 0 2 0 1   3 1 3 1 2 3  6 0 3 0 3 0 1   3 1 3 1 2 3  6 0 5 0 2 0 1   3 1 3 2 3 3  8 0 2 0 1 0 1 0 1   3 1 3 2 3 3  8 0 2 0 1 0 3 0 1   3 1 3 2 3 3  8 0 2 0 2 0 1 0 1   3 1 3 2 3 3  8 0 2 0 2 0 3 0 1   3 1 3 2 3 3  8 0 2 0 3 0 2 0 1   3 1 3 2 3 3  8 0 2 0 5 0 2 0 1   3 1 3 2 3 3  8 0 4 0 3 0 2 0 1   3 1 1 4  4 0 1 0 1   3 1 2 2 4  6 0 2 0 1 0 1   3 1 2 2 4  6 0 2 0 3 0 1   3 1 2 2 4  6 0 5 0 1 0 1   3 1 2 4 4  8 0 1 0 2 0 1 0 1   3 1 2 4 4  8 0 1 0 2 0 3 0 1   3 1 2 4 4  8 0 1 0 5 0 1 0 1   3 1 2 4 4  8 0 3 0 2 0 1 0 1   3 1 2 4 4  8 0 4 0 3 0 2 0 1   3 1 2 4 4  8 0 4 0 3 0 3 0 1   3 1 3 1 1 4  6 0 2 0 1 0 1   3 1 3 1 1 4  6 0 2 0 3 0 1   3 1 3 1 3 4  8 0 1 0 2 0 1 0 1   3 1 3 1 3 4  8 0 1 0 2 0 3 0 1   3 1 3 1 3 4  8 0 1 0 5 0 1 0 1   3 1 3 1 3 4  8 0 3 0 2 0 1 0 1   3 1 3 1 3 4  8 0 4 0 3 0 2 0 1   3 1 3 1 3 4  8 0 4 0 3 0 3 0 1   3 1 3 1 3 4  8 0 7 0 3 0 2 0 1   3 1 3 2 2 4  8 0 4 0 3 0 2 0 1   3 1 3 2 2 4  8 0 4 0 3 0 3 0 1   3 1 3 2 4 4  10 0 2 0 1 0 2 0 1 0 1   3 1 3 2 4 4  10 0 2 0 1 0 2 0 3 0 1   3 1 3 2 4 4  10 0 2 0 1 0 5 0 1 0 1   3 1 3 2 4 4  10 0 2 0 3 0 2 0 1 0 1   3 1 3 2 4 4  10 0 2 0 3 0 2 0 3 0 1   3 1 3 2 4 4  10 0 2 0 4 0 3 0 2 0 1   3 1 3 2 4 4  10 0 2 0 7 0 3 0 2 0 1   3 1 3 2 4 4  10 0 5 0 4 0 3 0 2 0 1   3 1 3 3 3 4  10 0 1 0 4 0 3 0 2 0 1"};

static vtkMomentInvariantData langbein3DMatrixOrder2 = {2, "", "87 3 2 1 2  2 0 1   3 2 2 2 2  4 0 2 0 1   3 2 2 2 2  4 0 3 0 1   3 2 3 2 2 2  6 0 2 0 2 0 1   3 2 3 2 2 2  6 0 3 0 2 0 1   3 2 4 2 2 2 2  8 0 2 0 2 0 2 0 1   3 2 2 3 3  6 0 1 0 1 0 1   3 2 2 3 3  6 0 1 0 2 0 1   3 2 2 3 3  6 0 1 0 3 0 1   3 2 2 3 3  6 0 2 0 1 0 1   3 2 2 3 3  6 0 2 0 2 0 1   3 2 2 3 3  6 0 3 0 1 0 1   3 2 2 3 3  6 0 3 0 2 0 1   3 2 2 3 3  6 0 3 0 3 0 1   3 2 2 3 3  6 0 4 0 2 0 1   3 2 2 3 3  6 0 4 0 3 0 1   3 2 2 3 3  6 0 5 0 3 0 1   3 2 3 2 3 3  8 0 2 0 1 0 1 0 1   3 2 3 2 3 3  8 0 2 0 1 0 2 0 1   3 2 3 2 3 3  8 0 2 0 1 0 3 0 1   3 2 3 2 3 3  8 0 2 0 2 0 1 0 1   3 2 3 2 3 3  8 0 2 0 2 0 2 0 1   3 2 3 2 3 3  8 0 2 0 2 0 3 0 1   3 2 3 2 3 3  8 0 2 0 3 0 1 0 1   3 2 3 2 3 3  8 0 2 0 3 0 2 0 1   3 2 3 2 3 3  8 0 2 0 3 0 3 0 1   3 2 3 2 3 3  8 0 2 0 4 0 1 0 1   3 2 3 2 3 3  8 0 2 0 4 0 2 0 1   3 2 3 2 3 3  8 0 2 0 4 0 3 0 1   3 2 3 2 3 3  8 0 2 0 5 0 1 0 1   3 2 3 2 3 3  8 0 2 0 5 0 2 0 1   3 2 3 2 3 3  8 0 3 0 1 0 1 0 1   3 2 3 2 3 3  8 0 3 0 1 0 2 0 1   3 2 1 4  4 0 1 0 1   3 2 1 4  4 0 2 0 1   3 2 2 2 4  6 0 2 0 1 0 1   3 2 2 2 4  6 0 2 0 2 0 1   3 2 2 2 4  6 0 2 0 3 0 1   3 2 2 2 4  6 0 4 0 1 0 1   3 2 2 2 4  6 0 4 0 3 0 1   3 2 2 2 4  6 0 5 0 1 0 1   3 2 2 2 4  6 0 5 0 3 0 1   3 2 2 4 4  8 0 1 0 2 0 1 0 1   3 2 2 4 4  8 0 1 0 2 0 2 0 1   3 2 2 4 4  8 0 1 0 2 0 3 0 1   3 2 2 4 4  8 0 1 0 4 0 1 0 1   3 2 2 4 4  8 0 1 0 4 0 3 0 1   3 2 2 4 4  8 0 1 0 5 0 1 0 1   3 2 2 4 4  8 0 1 0 5 0 3 0 1   3 2 2 4 4  8 0 2 0 2 0 1 0 1   3 2 2 4 4  8 0 2 0 2 0 2 0 1   3 2 2 4 4  8 0 2 0 2 0 3 0 1   3 2 2 4 4  8 0 2 0 4 0 1 0 1   3 2 2 4 4  8 0 2 0 5 0 1 0 1   3 2 2 4 4  8 0 3 0 2 0 1 0 1   3 2 2 4 4  8 0 3 0 2 0 2 0 1   3 2 2 4 4  8 0 4 0 3 0 1 0 1   3 2 2 4 4  8 0 4 0 3 0 2 0 1   3 2 2 4 4  8 0 4 0 3 0 3 0 1   3 2 2 4 4  8 0 4 0 4 0 2 0 1   3 2 2 4 4  8 0 4 0 4 0 3 0 1   3 2 3 2 2 4  8 0 2 0 2 0 1 0 1   3 2 3 2 2 4  8 0 2 0 2 0 2 0 1   3 2 3 2 2 4  8 0 2 0 2 0 3 0 1   3 2 3 2 2 4  8 0 3 0 2 0 1 0 1   3 2 3 2 2 4  8 0 3 0 2 0 2 0 1   3 2 3 2 2 4  8 0 3 0 2 0 3 0 1   3 2 3 2 2 4  8 0 4 0 2 0 1 0 1   3 2 3 2 2 4  8 0 4 0 3 0 2 0 1   3 2 3 2 2 4  8 0 4 0 3 0 3 0 1   3 2 3 2 2 4  8 0 4 0 4 0 2 0 1   3 2 3 2 4 4  10 0 2 0 1 0 2 0 1 0 1   3 2 3 2 4 4  10 0 2 0 1 0 2 0 2 0 1   3 2 3 2 4 4  10 0 2 0 1 0 2 0 3 0 1   3 2 3 2 4 4  10 0 2 0 1 0 4 0 1 0 1   3 2 3 2 4 4  10 0 2 0 1 0 4 0 3 0 1   3 2 3 2 4 4  10 0 2 0 1 0 5 0 1 0 1   3 2 3 2 4 4  10 0 2 0 2 0 2 0 1 0 1   3 2 3 2 4 4  10 0 2 0 2 0 2 0 2 0 1   3 2 3 2 4 4  10 0 2 0 2 0 2 0 3 0 1   3 2 3 2 4 4  10 0 2 0 2 0 4 0 1 0 1   3 2 3 2 4 4  10 0 2 0 2 0 4 0 3 0 1   3 2 3 2 4 4  10 0 2 0 3 0 2 0 1 0 1   3 2 3 2 4 4  10 0 2 0 3 0 2 0 2 0 1   3 2 3 2 4 4  10 0 2 0 4 0 3 0 2 0 1   3 2 3 2 4 4  10 0 2 0 6 0 3 0 2 0 1   3 2 3 2 4 4  10 0 2 0 7 0 3 0 2 0 1"};
#endif // __VTK_WRAP__
#endif //VTKMOMENTINVARIANTDATA_H
