/*=========================================================================

  Program:   ParaView
  Module:    vtkMergeArrays.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkMergeArrays.h"

#include "vtkCellData.h"
#include "vtkCommunicator.h"
#include "vtkDataArray.h"
#include "vtkDataSet.h"
#include "vtkFieldData.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMultiProcessController.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"

#include <cassert>

vtkStandardNewMacro(vtkMergeArrays);

//----------------------------------------------------------------------------
vtkMergeArrays::vtkMergeArrays()
{
}

//----------------------------------------------------------------------------
vtkMergeArrays::~vtkMergeArrays()
{
}

//----------------------------------------------------------------------------
bool vtkMergeArrays::GetOutputArrayName(
  vtkFieldData* arrays, const char* arrayName, int inputIndex, std::string& outputArrayName)
{
  if (arrays->GetArray(arrayName) == nullptr)
  {
    return false;
  }
  outputArrayName = std::string(arrayName) + "_input_" + std::to_string(inputIndex);
  return true;
}

//----------------------------------------------------------------------------
void vtkMergeArrays::MergeArrays(int inputIndex, vtkFieldData* inputFD, vtkFieldData* outputFD)
{
  if (inputFD == nullptr || outputFD == nullptr)
  {
    return;
  }

  std::string outputArrayName;
  int numArrays = inputFD->GetNumberOfArrays();
  for (int arrayIdx = 0; arrayIdx < numArrays; ++arrayIdx)
  {
    vtkDataArray* array = inputFD->GetArray(arrayIdx);
    if (this->GetOutputArrayName(outputFD, array->GetName(), inputIndex, outputArrayName))
    {
      vtkDataArray* newArray = array->NewInstance();
      newArray->ShallowCopy(array);
      newArray->SetName(outputArrayName.c_str());
      outputFD->AddArray(newArray);
      newArray->FastDelete();
    }
    else
    {
      outputFD->AddArray(array);
    }
  }
}

//----------------------------------------------------------------------------
int vtkMergeArrays::FillInputPortInformation(int vtkNotUsed(port), vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkDataSet");
  info->Append(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkGraph");
  info->Append(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkTable");
  info->Set(vtkAlgorithm::INPUT_IS_REPEATABLE(), 1);
  return 1;
}

//----------------------------------------------------------------------------
// Append data sets into single data set
int vtkMergeArrays::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  const int numInputs = inputVector[0]->GetNumberOfInformationObjects();
  if (numInputs < 1)
  {
    return 0;
  }

  // get the output and shallow copy input 0.
  vtkDataObject* output = vtkDataObject::GetData(outputVector, 0);
  output->ShallowCopy(vtkDataObject::GetData(inputVector[0], 0));
  if (numInputs < 2)
  {
    // nothing to do.
    return 1;
  }

  // check that the number of points and cells are the same for each process
  std::vector<int> checks;
  checks.reserve((numInputs-1) * vtkDataObject::NUMBER_OF_ATTRIBUTE_TYPES);
  for (int idx = 1; idx < numInputs; ++idx)
  {
    vtkDataObject* input = vtkDataObject::GetData(inputVector[0], idx);
    for (int attr=0; attr < vtkDataObject::NUMBER_OF_ATTRIBUTE_TYPES; ++attr)
    {
      checks.push_back(
          output->GetNumberOfElements(attr) == input->GetNumberOfElements(attr) ?
          0 : 1);
    }
  }
  assert(checks.size() > 0);

  std::vector<int> globalChecks(checks.size());
  vtkMultiProcessController::GetGlobalController()->AllReduce(
    &checks[0], &globalChecks[0], checks.size(), vtkCommunicator::MAX_OP);

  for (int idx = 1; idx < numInputs; ++idx)
  {
    vtkDataObject* input = vtkDataObject::GetData(inputVector[0], idx);
    for (int attr=0; attr < vtkDataObject::NUMBER_OF_ATTRIBUTE_TYPES; ++attr)
    {
      if (globalChecks[(idx - 1) * vtkDataObject::NUMBER_OF_ATTRIBUTE_TYPES + attr] == 0)
      {
        this->MergeArrays(idx, input->GetAttributesAsFieldData(attr),
            output->GetAttributesAsFieldData(attr));
      }
    }
  }

  return 1;
}

//----------------------------------------------------------------------------
void vtkMergeArrays::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}
