/*=========================================================================

  Program:   ParaView
  Module:    vtkPythonObservableWrapper.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPythonObservableWrapper.h"

#include "vtkObject.h"
#include "vtkObservableExamplePython.h"
#include "vtkPVApplication.h"
#include "vtkPythonRunLoop.h"
#include "vtkSMProxy.h"
#include "vtkSMSourceProxy.h"
#include "vtkSMTrace.h"
#include "vtkSmartPyObject.h"

namespace
{
bool CheckAndClearError()
{
  if (PyObject* exception = PyErr_Occurred())
  {
    PyErr_Print();
    PyErr_Clear();
    return true;
  }
  return false;
}

//----------------------------------------------------------------------------
template <typename T>
void PushToQueue(PyObject* queue, T value)
{
  (void)queue;
  (void)value;
  throw std::runtime_error("error pushing to iterator's queue");
}

template <>
void PushToQueue(PyObject* queue, int value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(queue, "put_nowait", "i", value) == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error pushing to iterator's queue");
  }
}
//----------------------------------------------------------------------------

template <typename T>
void SetFutureValue(PyObject* future, T value)
{
  (void)future;
  (void)value;
  throw std::runtime_error("error setting future's value");
}

template <>
void SetFutureValue(PyObject* future, bool value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(future, "set_result", "O", PyBool_FromLong(value)) == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error setting future's value");
  }
}

template <>
void SetFutureValue(PyObject* future, vtkTypeUInt32 value)
{
  vtkPythonScopeGilEnsurer gilEnsurer;
  if (PyObject_CallMethod(future, "set_result", "I", value) == nullptr)
  {
    CheckAndClearError();
    throw std::runtime_error("error setting future's value");
  }
}
}

//----------------------------------------------------------------------------
// TODO return None of failure ?
PyObject* vtkPythonObservableWrapper::GetFuture(
  vtkObject* object, std::string methodName, PyObject* args)
{
  PyObject* future = nullptr;
  // TODO with inheritance it becomes too cumbersome !
  if (auto* proxy = vtkSMProxy::SafeDownCast(object))
  {

    if (methodName == "UpdateInformation")
    {
      future = vtkPythonRunLoop::GetInstance()->CreateFuture();
      proxy->UpdateInformation().subscribe(
        [future](bool status) { SetFutureValue(future, status); });
    }
    else if (methodName == "ExecuteCommand")
    {
      assert(args);

      const char* name;
      vtkTypeUInt32 destinationMask = 0;

      if (PyArg_ParseTuple(args, "s|I", &name, &destinationMask) == 0)
      {
        CheckAndClearError();
        throw std::runtime_error("error unpacking arguments for ExecuteCommand");
      }

      future = vtkPythonRunLoop::GetInstance()->CreateFuture();
      if (destinationMask)
      {
        proxy->ExecuteCommand(name, destinationMask).subscribe([future](bool status) {
          SetFutureValue(future, status);
        });
      }
      else
      {
        proxy->ExecuteCommand(name).subscribe(
          [future](bool status) { SetFutureValue(future, status); });
      }
    }

    if (auto* proxy = vtkSMSourceProxy::SafeDownCast(object))
    {
      if (methodName == "UpdatePipeline")
      {
        future = vtkPythonRunLoop::GetInstance()->CreateFuture();
        proxy->UpdatePipeline(0.0).subscribe(
          [future](bool status) { SetFutureValue(future, status); });
      }
    }
  }
  else if (auto* app = vtkPVApplication::SafeDownCast(object))
  {
    if (methodName == "CreateBuiltinSession")
    {
      future = vtkPythonRunLoop::GetInstance()->CreateFuture();
      app->CreateBuiltinSession().subscribe(
        [future](vtkTypeUInt32 id) { SetFutureValue(future, id); });
    }
    else if (methodName == "CreateRemoteSession")
    {
      // TODO pass arguments
      abort();
    }
  }
  else
  {
    vtkLogF(ERROR, "Unknown class %s", object->GetClassName());
  }

  if (!future)
  {
    vtkLogF(ERROR, "Unknown methodname %s", methodName.c_str());
    abort();
  }

  return future;
}

//----------------------------------------------------------------------------
PyObject* vtkPythonObservableWrapper::GetIterator(
  vtkObject* object, std::string methodName, PyObject* args)
{
  vtkSmartPyObject iterator = nullptr;
  if (auto* proxy = vtkObservableExamplePython::SafeDownCast(object))
  {

    if (methodName == "GetObservable")
    {
      iterator = vtkPythonRunLoop::GetInstance()->CreateAsyncIterator();
      proxy->GetObservable().subscribe(

        // on_next
        [iterator](int value) {
          vtkPythonScopeGilEnsurer gilEnsurer;
          PyObject* queue = PyObject_CallMethod(iterator, "get_queue", "");
          if (queue == nullptr)
          {
            CheckAndClearError();
            throw std::runtime_error("error getting iterator's queue");
          }
          PushToQueue(queue, value);
        },

        // on_completed
        [iterator]() {
          vtkPythonScopeGilEnsurer gilEnsurer;
          PyObject* future = PyObject_CallMethod(iterator, "get_future", "");
          if (future == nullptr)
          {
            CheckAndClearError();
            throw std::runtime_error("error getting iterator's queue");
          }

          SetFutureValue(future, true);
        });
    }
  }
  else
  {
    vtkLogF(ERROR, "Unknown class %s", object->GetClassName());
  }

  return iterator;
}
