/*=========================================================================

   Program: ParaView
   Module:    pqSaveStateReaction.cxx

   Copyright (c) 2005,2006 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2.

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

========================================================================*/
#include "pqSaveStateReaction.h"

#include "pqActiveObjects.h"
#include "pqCoreUtilities.h"
#include "pqFileDialog.h"
#include "pqProxyWidgetDialog.h"
#include "pqServer.h"
#include "pqStandardRecentlyUsedResourceLoaderImplementation.h"
#include "vtkNew.h"
#include "vtkSMParaViewPipelineController.h"
#include "vtkSMPropertyHelper.h"
#include "vtkSMProxy.h"
#include "vtkSMSessionProxyManager.h"
#include "vtkSMTrace.h"
#include "vtkSmartPointer.h"

#include <QFile>
#include <QTextStream>
#include <QtDebug>

#include <cassert>

//-----------------------------------------------------------------------------
pqSaveStateReaction::pqSaveStateReaction(QAction* parentObject)
  : Superclass(parentObject)
{
  // save state enable state depends on whether we are connected to an active
  // server or not and whether
  pqActiveObjects* activeObjects = &pqActiveObjects::instance();
  QObject::connect(
    activeObjects, SIGNAL(serverChanged(pqServer*)), this, SLOT(updateEnableState()));
  this->updateEnableState();
}

//-----------------------------------------------------------------------------
void pqSaveStateReaction::updateEnableState()
{
  pqActiveObjects* activeObjects = &pqActiveObjects::instance();
  this->parentAction()->setEnabled(activeObjects->activeServer() != nullptr);
}

//-----------------------------------------------------------------------------
bool pqSaveStateReaction::saveState()
{
#if VTK_MODULE_ENABLE_ParaView_pqPython
  QString fileExt = tr("ParaView state file (*.pvsm);;Python state file (*.py);;All files (*)");
#else
  QString fileExt = tr("ParaView state file (*.pvsm);;All files (*)");
#endif

  // FIXME: ASYNC
  // Previously, for client-server file dialog, we'd send server == nullptr, but
  // that can't work anymore. We'll need to rework on file browsing works on
  // local file system
  pqServer* server = pqActiveObjects::instance().activeServer();
  pqFileDialog fileDialog(
    server, pqCoreUtilities::mainWidget(), tr("Save State File"), QString(), fileExt);

  fileDialog.setObjectName("FileSaveServerStateDialog");
  fileDialog.setFileMode(pqFileDialog::AnyFile);

  if (fileDialog.exec() == QDialog::Accepted)
  {
    QString selectedFile = fileDialog.getSelectedFiles()[0];
    if (selectedFile.endsWith(".py"))
    {
      pqSaveStateReaction::savePythonState(selectedFile);
    }
    else
    {
      pqSaveStateReaction::saveState(selectedFile);
    }
    return true;
  }
  return false;
}

//-----------------------------------------------------------------------------
bool pqSaveStateReaction::saveState(const QString& filename)
{
  pqServer* server = pqActiveObjects::instance().activeServer();
  if (!server)
  {
    qCritical() << "No active server to save state!";
    return false;
  }

  server->proxyManager()->SaveXMLState(qUtf8Printable(filename));
  // Add this to the list of recent server resources ...
  pqStandardRecentlyUsedResourceLoaderImplementation::addStateFileToRecentResources(
    server, filename);
  return true;
}

//-----------------------------------------------------------------------------
bool pqSaveStateReaction::savePythonState(const QString& filename)
{
#if VTK_MODULE_ENABLE_ParaView_pqPython
  vtkSMSessionProxyManager* pxm = pqActiveObjects::instance().proxyManager();
  assert(pxm);

  vtkSmartPointer<vtkSMProxy> options;
  options.TakeReference(pxm->NewProxy("pythontracing", "PythonStateOptions"));
  if (options.GetPointer() == nullptr)
  {
    return false;
  }

  vtkNew<vtkSMParaViewPipelineController> controller;
  controller->InitializeProxy(options);

  pqProxyWidgetDialog dialog(options);
  dialog.setWindowTitle("Python State Options");
  dialog.setObjectName("StateOptionsDialog");
  dialog.setApplyChangesImmediately(true);
  if (dialog.exec() != QDialog::Accepted)
  {
    return false;
  }

  std::string state = vtkSMTrace::GetState(options);
  if (state.empty())
  {
    qWarning("Empty state generated.");
    return false;
  }
  QFile file(filename);
  if (!file.open(QIODevice::WriteOnly | QIODevice::Text))
  {
    qWarning() << "Could not open file:" << filename;
    return false;
  }
  QTextStream out(&file);
  out << state.c_str();
  pqServer* server = pqActiveObjects::instance().activeServer();
  // Add this to the list of recent server resources ...
  pqStandardRecentlyUsedResourceLoaderImplementation::addStateFileToRecentResources(
    server, filename);
  return true;
#else
  qCritical() << "Failed to save '" << filename
              << "' since Python support in not enabled in this build.";
  return false;
#endif
}
