// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! Webhook listener
//!
//! This program listens over HTTP for JSON webhook events to pass along to a handler.
//!
//! See the [usage](usage.html) documentation for more.

#![warn(missing_docs)]

#[macro_use]
extern crate clap;

#[macro_use]
extern crate human_panic;

mod crates {
    pub extern crate clap;
    pub extern crate env_logger;
    pub extern crate futures;
    pub extern crate http;
    pub extern crate hyper;
    pub extern crate log;
    pub extern crate thiserror;
    pub extern crate webhook_router;

    #[cfg(feature = "sentry")]
    pub extern crate sentry;

    #[cfg(feature = "systemd")]
    pub extern crate systemd;
}

use std::error::Error;
use std::net::ToSocketAddrs;
use std::path::Path;
use std::sync::Arc;

use crates::clap::{App, Arg};
use crates::env_logger;
use crates::futures::{future, Future, Stream};
use crates::hyper::{Body, Client, Request, Server};
use crates::log::LevelFilter;
#[cfg(feature = "sentry")]
use crates::sentry;
#[cfg(feature = "systemd")]
use crates::systemd::journal::JournalLog;
use crates::thiserror::Error;
use crates::webhook_router::{Config, Router};

/// Errors which can occur when handling alternative commands.
#[derive(Debug, Error)]
pub enum ServiceError {
    /// Hyper had a problem.
    #[error("hyper error: {}", _0)]
    Hyper(#[from] hyper::Error),
    /// The other end returned an error.
    #[error("http error: {}", _0)]
    Http(#[from] http::Error),
    /// Multiple addresses were given.
    #[error("multiple socket addresses")]
    MultipleSocketAddrs,
    /// No addresses were given.
    #[error("no socket addresses")]
    NoSocketAddrs,
}

/// Run on an address based on a configuration at the given path.
fn run_from_config(address: &str, config_path: &Path) -> Result<(), Box<dyn Error>> {
    let router = Arc::new(Router::new(config_path)?);
    let new_service = move || {
        let router = router.clone();

        hyper::service::service_fn(move |req: Request<Body>| {
            let (parts, body) = req.into_parts();
            let router = router.clone();

            body.concat2()
                .or_else(|source| future::err(ServiceError::from(source)))
                .and_then(move |chunk| {
                    let req = Request::from_parts(parts, chunk.iter().copied().collect());
                    router
                        .handle(&req)
                        .map(|rsp| rsp.map(Body::from))
                        .map_err(ServiceError::from)
                })
        })
    };

    let socket_addrs = address.to_socket_addrs()?.collect::<Vec<_>>();
    if socket_addrs.len() > 1 {
        return Err(ServiceError::MultipleSocketAddrs.into());
    }
    let socket_addr = socket_addrs
        .into_iter()
        .next()
        .ok_or(ServiceError::NoSocketAddrs)?;
    let server = Server::bind(&socket_addr)
        .serve(new_service)
        .map_err(|err| eprintln!("{:?}", err));

    hyper::rt::run(server);

    Ok(())
}

#[derive(Debug, Error)]
enum LogError {
    #[cfg(feature = "sentry")]
    #[error("`sentry` requires `--sentry-endpoint=`")]
    SentryEndpointMissing,
    #[cfg(not(feature = "sentry"))]
    #[error("logging: `sentry` support is not available")]
    NoSentry,
    #[cfg(not(feature = "systemd"))]
    #[error("logging: `systemd` support is not available")]
    NoSystemd,

    #[error("unknown logger: {}", _0)]
    UnknownLogger(String),
}

enum Logger {
    #[cfg(feature = "systemd")]
    Systemd,
    #[cfg(feature = "sentry")]
    Sentry(sentry::internals::ClientInitGuard),
    Env,
}

#[derive(Debug, Error)]
enum RunError {
    #[error("reload HTTP error: {}", _0)]
    ReloadHttp(#[source] hyper::Error),
    #[error("reload error: {}", _0)]
    Reload(String),
}

/// A `main` function which supports `try!`.
fn try_main() -> Result<(), Box<dyn Error>> {
    setup_panic!();

    let matches = App::new("webhook-listen")
        .version(crate_version!())
        .author("Ben Boeckel <ben.boeckel@kitware.com>")
        .about("Listen over HTTP for JSON webhook events to pass along to a handler")
        .arg(
            Arg::with_name("ADDRESS")
                .short("a")
                .long("address")
                .help("The address to listen on")
                .required_unless("VERIFY")
                .value_name("ADDRESS")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("CONFIG")
                .short("c")
                .long("config")
                .help("Path to the configuration file")
                .required_unless("RELOAD")
                .value_name("FILE")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("DEBUG")
                .short("d")
                .long("debug")
                .help("Increase verbosity")
                .multiple(true),
        )
        .arg(
            Arg::with_name("LOGGER")
                .short("l")
                .long("logger")
                .default_value("env")
                .possible_values(&[
                    "env",
                    #[cfg(feature = "systemd")]
                    "systemd",
                    #[cfg(feature = "sentry")]
                    "sentry",
                ])
                .help("Logging backend")
                .value_name("LOGGER")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("SENTRY_ENDPOINT")
                .long("sentry-endpoint")
                .help("Sentry logging endpoint")
                .hidden(cfg!(not(feature = "sentry")))
                .value_name("ENDPOINT")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("VERIFY")
                .short("v")
                .long("verify")
                .help("Check the configuration file and exit"),
        )
        .arg(
            Arg::with_name("RELOAD")
                .short("r")
                .long("reload")
                .help("Reload the configuration"),
        )
        .get_matches();

    let log_level = match matches.occurrences_of("DEBUG") {
        0 => LevelFilter::Error,
        1 => LevelFilter::Warn,
        2 => LevelFilter::Info,
        3 => LevelFilter::Debug,
        _ => LevelFilter::Trace,
    };

    let _logger = match matches
        .value_of("LOGGER")
        .expect("logger should have a value")
    {
        "env" => {
            env_logger::Builder::new().filter(None, log_level).init();
            Logger::Env
        },

        #[cfg(feature = "systemd")]
        "systemd" => {
            JournalLog::init()?;
            Logger::Systemd
        },
        #[cfg(not(feature = "systemd"))]
        "systemd" => {
            return Err(LogError::NoSystemd.into());
        },

        #[cfg(feature = "sentry")]
        "sentry" => {
            if let Some(endpoint) = matches.value_of("SENTRY_ENDPOINT") {
                Logger::Sentry(sentry::init(endpoint))
            } else {
                return Err(LogError::SentryEndpointMissing.into());
            }
        },
        #[cfg(not(feature = "sentry"))]
        "sentry" => {
            return Err(LogError::NoSentry.into());
        },

        logger => {
            return Err(LogError::UnknownLogger(logger.into()).into());
        },
    };

    log::set_max_level(log_level);

    let config_path = Path::new(
        matches
            .value_of("CONFIG")
            .expect("the configuration option is required"),
    );

    if matches.is_present("VERIFY") {
        Config::from_path(&config_path)?;
    } else if matches.is_present("RELOAD") {
        let address = matches
            .value_of("ADDRESS")
            .expect("the address option is required");
        let url = format!("http://{}/__reload", address);
        let req = Request::put(url).body(Body::empty())?;
        let client = Client::new();

        hyper::rt::run(
            client
                .request(req)
                .map_err(RunError::ReloadHttp)
                .and_then(|rsp| {
                    if rsp.status().is_success() {
                        Box::new(future::ok(())) as Box<dyn Future<Item = _, Error = _> + Send>
                    } else {
                        Box::new(
                            rsp.into_body()
                                .map_err(RunError::ReloadHttp)
                                .concat2()
                                .and_then(move |chunk| {
                                    let data = chunk.iter().copied().collect::<Vec<_>>();
                                    let err =
                                        RunError::Reload(String::from_utf8_lossy(&data).into());
                                    future::err(err)
                                }),
                        )
                    }
                })
                .map_err(|err| eprintln!("{:?}", err)),
        );
    } else {
        run_from_config(
            matches
                .value_of("ADDRESS")
                .expect("the address option is required"),
            config_path,
        )?;
    }

    Ok(())
}

/// The entry point.
///
/// Wraps around `try_main` to panic on errors.
fn main() {
    if let Err(err) = try_main() {
        panic!("{:?}", err);
    }
}
