// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::io::Result;
use std::path::Path;

use futures::Stream;
use inotify::{EventOwned, Inotify, WatchMask};

/// A watcher of a directory for new files.
pub(crate) struct Watcher {
    /// The internal inotify watcher.
    ino: Inotify,
}

impl Watcher {
    /// Watch a given directory.
    pub(crate) fn new(path: &Path) -> Result<Self> {
        let ino = Inotify::init()?;
        ino.watches().add(
            path,
            WatchMask::CLOSE_WRITE | WatchMask::MOVED_TO | WatchMask::ONLYDIR,
        )?;

        Ok(Self {
            ino,
        })
    }

    /// Create a stream of events.
    pub(crate) fn events(self) -> Result<impl Stream<Item = Result<EventOwned>>> {
        self.ino.into_event_stream([0; 1024])
    }
}
