// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::{self, NameOrId};
use crate::api::endpoint_prelude::*;

/// Get a single file from repository.
#[derive(Debug, Builder, Clone)]
pub struct File<'a> {
    /// The project to get a file within.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The path to the file in the repository.
    ///
    /// This is automatically escaped as needed.
    #[builder(setter(into))]
    file_path: Cow<'a, str>,
    /// The ref to get a file from.
    #[builder(setter(into))]
    ref_: Cow<'a, str>,
}

impl<'a> File<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> FileBuilder<'a> {
        FileBuilder::default()
    }
}

impl Endpoint for File<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/repository/files/{}",
            self.project,
            common::path_escaped(&self.file_path),
        )
        .into()
    }

    fn parameters(&self) -> QueryParams {
        let mut params = QueryParams::default();

        params.push("ref", &self.ref_);

        params
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::repository::files::{File, FileBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn all_parameters_are_needed() {
        let err = File::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, FileBuilderError, "project");
    }

    #[test]
    fn project_is_required() {
        let err = File::builder()
            .file_path("new/file")
            .ref_("master")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, FileBuilderError, "project");
    }

    #[test]
    fn file_path_is_required() {
        let err = File::builder()
            .project(1)
            .ref_("master")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, FileBuilderError, "file_path");
    }

    #[test]
    fn ref_is_required() {
        let err = File::builder()
            .project(1)
            .file_path("new/file")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, FileBuilderError, "ref_");
    }

    #[test]
    fn sufficient_parameters() {
        File::builder()
            .project(1)
            .file_path("new/file")
            .ref_("master")
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/simple%2Fproject/repository/files/path%2Fto%2Ffile")
            .add_query_params(&[("ref", "branch")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = File::builder()
            .project("simple/project")
            .file_path("path/to/file")
            .ref_("branch")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
