// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::any;
use std::error::Error;

use thiserror::Error;

use crate::api::{PaginationError, UrlBase};

/// Errors which may occur when creating form data.
#[derive(Debug, Error)]
#[non_exhaustive]
pub enum BodyError {
    /// Body data could not be serialized from form parameters.
    #[error("failed to URL encode form parameters: {}", source)]
    UrlEncoded {
        /// The source of the error.
        #[from]
        source: serde_urlencoded::ser::Error,
    },
    /// Body data could not be serialized to JSON from form parameters.
    #[error("failed to JSON encode form parameters: {}", source)]
    JsonEncoded {
        /// The source of the error.
        #[from]
        source: serde_json::Error,
    },
}

/// Errors which may occur when using API endpoints.
#[derive(Debug, Error)]
#[non_exhaustive]
pub enum ApiError<E>
where
    E: Error + Send + Sync + 'static,
{
    /// The client encountered an error.
    #[error("client error: {}", source)]
    Client {
        /// The client error.
        source: E,
    },
    /// Authentication failed.
    #[error("failed to authenticate: {}", source)]
    Auth {
        /// The source of the error.
        #[from]
        source: crate::AuthError,
    },
    /// The URL failed to parse.
    #[error("failed to parse url: {}", source)]
    UrlParse {
        /// The source of the error.
        #[from]
        source: url::ParseError,
    },
    /// Body data could not be created.
    #[error("failed to create form data: {}", source)]
    Body {
        /// The source of the error.
        #[from]
        source: BodyError,
    },
    /// JSON deserialization from GitLab failed.
    #[error("could not parse JSON response: {}", source)]
    Json {
        /// The source of the error.
        #[from]
        source: serde_json::Error,
    },
    /// The resource has been moved permanently.
    #[error("moved permanently to: {}", location.as_ref().map(AsRef::as_ref).unwrap_or("<UNKNOWN>"))]
    MovedPermanently {
        /// The new location for the resource.
        location: Option<String>,
    },
    /// GitLab returned an error message.
    #[error("gitlab server error: {}", msg)]
    #[deprecated(since = "0.1705.2", note = "Use `GitlabWithStatus` instead")]
    Gitlab {
        /// The error message from GitLab.
        msg: String,
    },
    /// GitLab returned an error without JSON information.
    #[error("gitlab internal server error {}", status)]
    GitlabService {
        /// The status code for the return.
        status: http::StatusCode,
        /// The error data from GitLab.
        data: Vec<u8>,
    },
    /// GitLab returned an error object.
    #[error("gitlab server error: {:?}", obj)]
    #[deprecated(since = "0.1705.2", note = "Use `GitlabObjectWithStatus` instead")]
    GitlabObject {
        /// The error object from GitLab.
        obj: serde_json::Value,
    },
    /// GitLab returned an HTTP error with JSON we did not recognize.
    #[error("gitlab server error: {:?}", obj)]
    #[deprecated(
        since = "0.1705.2",
        note = "Use `GitlabUnrecognizedWithStatus` instead"
    )]
    GitlabUnrecognized {
        /// The full object from GitLab.
        obj: serde_json::Value,
    },
    /// Failed to parse an expected data type from JSON.
    #[error("could not parse {} data from JSON: {}", typename, source)]
    DataType {
        /// The source of the error.
        source: serde_json::Error,
        /// The name of the type that could not be deserialized.
        typename: &'static str,
    },
    /// An error with pagination occurred.
    #[error("failed to handle for pagination: {}", source)]
    Pagination {
        /// The source of the error.
        #[from]
        source: PaginationError,
    },
    /// The client does not understand how to use an endpoint for the given URL base.
    #[error("unsupported URL base: {:?}", url_base)]
    UnsupportedUrlBase {
        /// The URL base that is not supported.
        url_base: UrlBase,
    },
    /// GitLab returned an error message with an HTTP error.
    #[error("gitlab server error ({}): {}", status, msg)]
    GitlabWithStatus {
        /// The HTTP status code.
        status: http::StatusCode,
        /// The error message from GitLab.
        msg: String,
    },
    /// GitLab returned an error object with an HTTP error.
    #[error("gitlab server error ({}): {:?}", status, obj)]
    GitlabObjectWithStatus {
        /// The HTTP status code.
        status: http::StatusCode,
        /// The error object from GitLab.
        obj: serde_json::Value,
    },
    /// GitLab returned an HTTP error with JSON we did not recognize.
    #[error("gitlab server error ({}): {:?}", status, obj)]
    GitlabUnrecognizedWithStatus {
        /// The HTTP status code.
        status: http::StatusCode,
        /// The full object from GitLab.
        obj: serde_json::Value,
    },
}

impl<E> ApiError<E>
where
    E: Error + Send + Sync + 'static,
{
    /// Create an API error in a client error.
    pub fn client(source: E) -> Self {
        ApiError::Client {
            source,
        }
    }

    /// Wrap a client error in another wrapper.
    pub fn map_client<F, W>(self, f: F) -> ApiError<W>
    where
        F: FnOnce(E) -> W,
        W: Error + Send + Sync + 'static,
    {
        match self {
            Self::Client {
                source,
            } => ApiError::client(f(source)),
            Self::UrlParse {
                source,
            } => {
                ApiError::UrlParse {
                    source,
                }
            },
            Self::Auth {
                source,
            } => {
                ApiError::Auth {
                    source,
                }
            },
            Self::Body {
                source,
            } => {
                ApiError::Body {
                    source,
                }
            },
            Self::Json {
                source,
            } => {
                ApiError::Json {
                    source,
                }
            },
            Self::MovedPermanently {
                location,
            } => {
                ApiError::MovedPermanently {
                    location,
                }
            },
            #[allow(deprecated)]
            Self::Gitlab {
                msg,
            } =>
            {
                #[allow(deprecated)]
                ApiError::Gitlab {
                    msg,
                }
            },
            Self::GitlabWithStatus {
                status,
                msg,
            } => {
                ApiError::GitlabWithStatus {
                    status,
                    msg,
                }
            },
            Self::GitlabService {
                status,
                data,
            } => {
                ApiError::GitlabService {
                    status,
                    data,
                }
            },
            #[allow(deprecated)]
            Self::GitlabObject {
                obj,
            } =>
            {
                #[allow(deprecated)]
                ApiError::GitlabObject {
                    obj,
                }
            },
            Self::GitlabObjectWithStatus {
                status,
                obj,
            } => {
                ApiError::GitlabObjectWithStatus {
                    status,
                    obj,
                }
            },
            #[allow(deprecated)]
            Self::GitlabUnrecognized {
                obj,
            } =>
            {
                #[allow(deprecated)]
                ApiError::GitlabUnrecognized {
                    obj,
                }
            },
            Self::GitlabUnrecognizedWithStatus {
                status,
                obj,
            } => {
                ApiError::GitlabUnrecognizedWithStatus {
                    status,
                    obj,
                }
            },
            Self::DataType {
                source,
                typename,
            } => {
                ApiError::DataType {
                    source,
                    typename,
                }
            },
            Self::Pagination {
                source,
            } => {
                ApiError::Pagination {
                    source,
                }
            },
            Self::UnsupportedUrlBase {
                url_base,
            } => {
                ApiError::UnsupportedUrlBase {
                    url_base,
                }
            },
        }
    }

    pub(crate) fn moved_permanently(raw_location: Option<&http::HeaderValue>) -> Self {
        let location = raw_location.map(|v| String::from_utf8_lossy(v.as_bytes()).into());
        Self::MovedPermanently {
            location,
        }
    }

    pub(crate) fn server_error(status: http::StatusCode, body: &bytes::Bytes) -> Self {
        Self::GitlabService {
            status,
            data: body.into_iter().copied().collect(),
        }
    }

    pub(crate) fn from_gitlab_with_status(
        status: http::StatusCode,
        value: serde_json::Value,
    ) -> Self {
        let error_value = value
            .pointer("/message")
            .or_else(|| value.pointer("/error"));

        if let Some(error_value) = error_value {
            if let Some(msg) = error_value.as_str() {
                ApiError::GitlabWithStatus {
                    status,
                    msg: msg.into(),
                }
            } else {
                ApiError::GitlabObjectWithStatus {
                    status,
                    obj: error_value.clone(),
                }
            }
        } else {
            ApiError::GitlabUnrecognizedWithStatus {
                status,
                obj: value,
            }
        }
    }

    pub(crate) fn data_type<T>(source: serde_json::Error) -> Self {
        ApiError::DataType {
            source,
            typename: any::type_name::<T>(),
        }
    }

    pub(crate) fn unsupported_url_base(url_base: UrlBase) -> Self {
        Self::UnsupportedUrlBase {
            url_base,
        }
    }
}

#[cfg(test)]
mod tests {
    use serde_json::json;
    use thiserror::Error;

    use crate::api::ApiError;

    #[derive(Debug, Error)]
    #[error("my error")]
    enum MyError {}

    #[test]
    fn gitlab_error_error() {
        let obj = json!({
            "error": "error contents",
        });

        let expected_status = http::StatusCode::NOT_FOUND;
        let err: ApiError<MyError> = ApiError::from_gitlab_with_status(expected_status, obj);
        if let ApiError::GitlabWithStatus {
            status,
            msg,
        } = err
        {
            assert_eq!(status, expected_status);
            assert_eq!(msg, "error contents");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn gitlab_error_message_string() {
        let obj = json!({
            "message": "error contents",
        });

        let expected_status = http::StatusCode::NOT_FOUND;
        let err: ApiError<MyError> = ApiError::from_gitlab_with_status(expected_status, obj);
        if let ApiError::GitlabWithStatus {
            status,
            msg,
        } = err
        {
            assert_eq!(status, expected_status);
            assert_eq!(msg, "error contents");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn gitlab_error_message_object() {
        let err_obj = json!({
            "blah": "foo",
        });
        let obj = json!({
            "message": err_obj,
        });

        let expected_status = http::StatusCode::NOT_FOUND;
        let err: ApiError<MyError> = ApiError::from_gitlab_with_status(expected_status, obj);
        if let ApiError::GitlabObjectWithStatus {
            status,
            obj,
        } = err
        {
            assert_eq!(status, expected_status);
            assert_eq!(obj, err_obj);
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn gitlab_error_message_unrecognized() {
        let err_obj = json!({
            "some_weird_key": "an even weirder value",
        });

        let expected_status = http::StatusCode::NOT_FOUND;
        let err: ApiError<MyError> =
            ApiError::from_gitlab_with_status(expected_status, err_obj.clone());
        if let ApiError::GitlabUnrecognizedWithStatus {
            status,
            obj,
        } = err
        {
            assert_eq!(status, expected_status);
            assert_eq!(obj, err_obj);
        } else {
            panic!("unexpected error: {}", err);
        }
    }
}
