// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::endpoint_prelude::*;

/// Query for details of a certain container registry repository.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct RepositoryDetails {
    /// The repository for which to query details.
    repository_id: u64,
    /// When true, includes an array of "tags" in the response.
    #[builder(default)]
    tags: Option<bool>,
    /// When true, includes "tags_count" in the response.
    #[builder(default)]
    tags_count: Option<bool>,
    /// When true, includes "size" in the response.
    ///
    /// This is the deduplicated size of all images within the repository.
    #[builder(default)]
    size: Option<bool>,
}

impl RepositoryDetails {
    /// Create a builder for the endpoint.
    pub fn builder() -> RepositoryDetailsBuilder {
        RepositoryDetailsBuilder::default()
    }
}

impl Endpoint for RepositoryDetails {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("registry/repositories/{}", self.repository_id).into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params
            .push_opt("tags", self.tags)
            .push_opt("tags_count", self.tags_count)
            .push_opt("size", self.size);

        params
    }
}

#[cfg(test)]
mod tests {
    use crate::api::registry::repositories::{RepositoryDetails, RepositoryDetailsBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn repository_id_is_necessary() {
        let err = RepositoryDetails::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, RepositoryDetailsBuilderError, "repository_id");
    }

    #[test]
    fn repository_id_is_sufficient() {
        RepositoryDetails::builder()
            .repository_id(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("registry/repositories/42")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = RepositoryDetails::builder()
            .repository_id(42)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_tags() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("registry/repositories/42")
            .add_query_params(&[("tags", "false")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = RepositoryDetails::builder()
            .repository_id(42)
            .tags(false)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_tags_count() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("registry/repositories/42")
            .add_query_params(&[("tags_count", "true")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = RepositoryDetails::builder()
            .repository_id(42)
            .tags_count(true)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_size() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("registry/repositories/42")
            .add_query_params(&[("size", "false")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = RepositoryDetails::builder()
            .repository_id(42)
            .size(false)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
