// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Unstars a given project.
#[derive(Debug, Builder, Clone)]
pub struct UnstarProject<'a> {
    /// The project to unstar.
    #[builder(setter(into))]
    project: NameOrId<'a>,
}

impl<'a> UnstarProject<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> UnstarProjectBuilder<'a> {
        UnstarProjectBuilder::default()
    }
}

impl<'a> Endpoint for UnstarProject<'a> {
    fn method(&self) -> Method {
        Method::POST
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/unstar", self.project).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::star::{UnstarProject, UnstarProjectBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_necessary() {
        let err = UnstarProject::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, UnstarProjectBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        UnstarProject::builder().project("project").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("projects/project%2Fsubproject/unstar")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = UnstarProject::builder()
            .project("project/subproject")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
