// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Query a specific project release by tag name.
#[derive(Debug, Clone, Builder)]
#[builder(setter(strip_option))]
pub struct ProjectReleaseByTag<'a> {
    /// The project to query for release.
    #[builder(setter(into))]
    project: NameOrId<'a>,

    /// Gets a release for a specific tag
    #[builder(setter(into))]
    tag: Cow<'a, str>,

    /// Whether to include an HTML render of the description or not.
    #[builder(default)]
    include_html_description: Option<bool>,
}

impl<'a> ProjectReleaseByTag<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> ProjectReleaseByTagBuilder<'a> {
        ProjectReleaseByTagBuilder::default()
    }
}

impl Endpoint for ProjectReleaseByTag<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/releases/{}", self.project, self.tag).into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params.push_opt("include_html_description", self.include_html_description);

        params
    }
}

#[cfg(test)]
mod tests {
    use crate::api::projects::releases::release_by_tag::{
        ProjectReleaseByTag, ProjectReleaseByTagBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_needed() {
        let err = ProjectReleaseByTag::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, ProjectReleaseByTagBuilderError, "project");
    }

    #[test]
    fn tag_is_needed() {
        let err = ProjectReleaseByTag::builder()
            .project(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, ProjectReleaseByTagBuilderError, "tag");
    }

    #[test]
    fn project_and_tag_is_sufficient() {
        ProjectReleaseByTag::builder()
            .project(1)
            .tag("v1.0.0")
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/group%2Fproject/releases/v1.0.0")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = ProjectReleaseByTag::builder()
            .project("group/project")
            .tag("v1.0.0")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_include_html_description() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/project/releases/v1.0.0")
            .add_query_params(&[("include_html_description", "false")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = ProjectReleaseByTag::builder()
            .project("project")
            .tag("v1.0.0")
            .include_html_description(false)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
