// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Remove Pages.
#[derive(Debug, Builder, Clone)]
pub struct UnpublishPages<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,
}

impl<'a> UnpublishPages<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> UnpublishPagesBuilder<'a> {
        UnpublishPagesBuilder::default()
    }
}

impl Endpoint for UnpublishPages<'_> {
    fn method(&self) -> Method {
        Method::DELETE
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/pages", self.project).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::pages::{UnpublishPages, UnpublishPagesBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = UnpublishPages::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, UnpublishPagesBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        UnpublishPages::builder().project("blah").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/blah/pages")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = UnpublishPages::builder().project("blah").build().unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
