// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Keep job artifacts.
#[derive(Debug, Builder, Clone)]
pub struct KeepJobArtifacts<'a> {
    /// The project to keep artifacts for.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The job to keep artifacts for.
    job: u64,
}

impl<'a> KeepJobArtifacts<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> KeepJobArtifactsBuilder<'a> {
        KeepJobArtifactsBuilder::default()
    }
}

impl Endpoint for KeepJobArtifacts<'_> {
    fn method(&self) -> Method {
        Method::POST
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/jobs/{}/artifacts/keep", self.project, self.job).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::jobs::artifacts::{KeepJobArtifacts, KeepJobArtifactsBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = KeepJobArtifacts::builder().job(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, KeepJobArtifactsBuilderError, "project");
    }

    #[test]
    fn job_is_required() {
        let err = KeepJobArtifacts::builder().project(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, KeepJobArtifactsBuilderError, "job");
    }

    #[test]
    fn project_and_job_are_sufficient() {
        KeepJobArtifacts::builder()
            .project(1)
            .job(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts/keep")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = KeepJobArtifacts::builder()
            .project("example/repo")
            .job(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
