// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Download job artifacts.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct DownloadJobArtifacts<'a> {
    /// The project to download artifacts from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The job to download artifacts from.
    job: u64,
    /// The job token to authenticate with.
    ///
    /// For use with multi-project pipelines.
    #[builder(setter(into), default)]
    job_token: Option<Cow<'a, str>>,
}

impl<'a> DownloadJobArtifacts<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DownloadJobArtifactsBuilder<'a> {
        DownloadJobArtifactsBuilder::default()
    }
}

impl Endpoint for DownloadJobArtifacts<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/jobs/{}/artifacts", self.project, self.job).into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params.push_opt("job_token", self.job_token.as_ref());

        params
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::jobs::artifacts::{
        DownloadJobArtifacts, DownloadJobArtifactsBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DownloadJobArtifacts::builder().job(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, DownloadJobArtifactsBuilderError, "project");
    }

    #[test]
    fn job_is_required() {
        let err = DownloadJobArtifacts::builder()
            .project(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadJobArtifactsBuilderError, "job");
    }

    #[test]
    fn project_and_job_are_sufficient() {
        DownloadJobArtifacts::builder()
            .project(1)
            .job(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadJobArtifacts::builder()
            .project("example/repo")
            .job(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_job_token() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts")
            .add_query_params(&[("job_token", "TOKEN")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadJobArtifacts::builder()
            .project("example/repo")
            .job(1)
            .job_token("TOKEN")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
