// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Add a project to the 'CI/CD job token inbound allowlist' of a project.
#[derive(Debug, Builder, Clone)]
pub struct AllowJobTokenProject<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,

    /// The ID of the project added to the 'CI/CD job token inbound allowlist'.
    target_project_id: u64,
}

impl<'a> AllowJobTokenProject<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> AllowJobTokenProjectBuilder<'a> {
        AllowJobTokenProjectBuilder::default()
    }
}

impl Endpoint for AllowJobTokenProject<'_> {
    fn method(&self) -> Method {
        Method::POST
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/job_token_scope/allowlist", self.project).into()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        let mut params = FormParams::default();

        params.push("target_project_id", self.target_project_id);

        params.into_body()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::job_token_scopes::{
        AllowJobTokenProject, AllowJobTokenProjectBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = AllowJobTokenProject::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, AllowJobTokenProjectBuilderError, "project");
    }

    #[test]
    fn target_is_required() {
        let err = AllowJobTokenProject::builder()
            .project("blah")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(
            err,
            AllowJobTokenProjectBuilderError,
            "target_project_id",
        );
    }

    #[test]
    fn project_and_target_is_sufficient() {
        AllowJobTokenProject::builder()
            .project("blah")
            .target_project_id(42)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("projects/blah/job_token_scope/allowlist")
            .content_type("application/x-www-form-urlencoded")
            .body_str("target_project_id=42")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AllowJobTokenProject::builder()
            .project("blah")
            .target_project_id(42)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
