// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Delete all job artifacts on a project.
#[derive(Debug, Builder, Clone)]
pub struct DeleteProjectArtifacts<'a> {
    /// The project to delete artifacts from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
}

impl<'a> DeleteProjectArtifacts<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DeleteProjectArtifactsBuilder<'a> {
        DeleteProjectArtifactsBuilder::default()
    }
}

impl Endpoint for DeleteProjectArtifacts<'_> {
    fn method(&self) -> Method {
        Method::DELETE
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/artifacts", self.project).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::artifacts::{
        DeleteProjectArtifacts, DeleteProjectArtifactsBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DeleteProjectArtifacts::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, DeleteProjectArtifactsBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        DeleteProjectArtifacts::builder()
            .project(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/example%2Frepo/artifacts")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteProjectArtifacts::builder()
            .project("example/repo")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
