// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::borrow::Cow;

use async_trait::async_trait;
use http::{self, header, Method, Request};
use reqwest::Url;
use serde::de::DeserializeOwned;

use crate::api::{
    query, ApiError, AsyncClient, AsyncQuery, BodyError, Client, Query, QueryParams, RestClient,
};

/// URL bases for endpoints.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
#[non_exhaustive]
pub enum UrlBase {
    /// An endpoint uses the API v4 URL prefix.
    ApiV4,
    /// An endpoint acts on the instance itself.
    Instance,
}

impl UrlBase {
    /// Get the endpoint for a given URL base.
    pub fn endpoint_for<C>(&self, client: &C, endpoint: &str) -> Result<Url, ApiError<C::Error>>
    where
        C: RestClient,
    {
        match self {
            UrlBase::ApiV4 => client.rest_endpoint(endpoint),
            UrlBase::Instance => client.instance_endpoint(endpoint),
        }
    }
}

/// A trait for providing the necessary information for a single REST API endpoint.
pub trait Endpoint {
    /// The HTTP method to use for the endpoint.
    fn method(&self) -> Method;
    /// The path to the endpoint.
    fn endpoint(&self) -> Cow<'static, str>;

    /// The URL base of the API endpoint.
    fn url_base(&self) -> UrlBase {
        UrlBase::ApiV4
    }

    /// Query parameters for the endpoint.
    fn parameters(&self) -> QueryParams<'_> {
        QueryParams::default()
    }

    /// The body for the endpoint.
    ///
    /// Returns the `Content-Encoding` header for the data as well as the data itself.
    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        Ok(None)
    }
}

impl<E> Endpoint for &E
where
    E: Endpoint,
{
    fn method(&self) -> Method {
        (*self).method()
    }

    fn endpoint(&self) -> Cow<'static, str> {
        (*self).endpoint()
    }

    fn url_base(&self) -> UrlBase {
        (*self).url_base()
    }

    fn parameters(&self) -> QueryParams<'_> {
        (*self).parameters()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        (*self).body()
    }
}

impl<E, T, C> Query<T, C> for E
where
    E: Endpoint,
    T: DeserializeOwned,
    C: Client,
{
    fn query(&self, client: &C) -> Result<T, ApiError<C::Error>> {
        let mut url = self.url_base().endpoint_for(client, &self.endpoint())?;
        self.parameters().add_to_url(&mut url);

        let req = Request::builder()
            .method(self.method())
            .uri(query::url_to_http_uri(url));
        let (req, data) = if let Some((mime, data)) = self.body()? {
            let req = req.header(header::CONTENT_TYPE, mime);
            (req, data)
        } else {
            (req, Vec::new())
        };
        let rsp = client.rest(req, data)?;
        let status = rsp.status();
        if status == http::StatusCode::MOVED_PERMANENTLY {
            return Err(ApiError::moved_permanently(
                rsp.headers().get(http::header::LOCATION),
            ));
        } else if status == http::StatusCode::TOO_MANY_REQUESTS {
            return Err(ApiError::from_gitlab_rate_limit(rsp.headers()));
        }
        let v = if let Ok(v) = serde_json::from_slice(rsp.body()) {
            v
        } else {
            return Err(ApiError::server_error(status, rsp.body()));
        };
        if !status.is_success() {
            return Err(ApiError::from_gitlab_with_status(status, v));
        }

        serde_json::from_value::<T>(v).map_err(ApiError::data_type::<T>)
    }
}

#[async_trait]
impl<E, T, C> AsyncQuery<T, C> for E
where
    E: Endpoint + Sync,
    T: DeserializeOwned + 'static,
    C: AsyncClient + Sync,
{
    async fn query_async(&self, client: &C) -> Result<T, ApiError<C::Error>> {
        let mut url = self.url_base().endpoint_for(client, &self.endpoint())?;
        self.parameters().add_to_url(&mut url);

        let req = Request::builder()
            .method(self.method())
            .uri(query::url_to_http_uri(url));
        let (req, data) = if let Some((mime, data)) = self.body()? {
            let req = req.header(header::CONTENT_TYPE, mime);
            (req, data)
        } else {
            (req, Vec::new())
        };
        let rsp = client.rest_async(req, data).await?;
        let status = rsp.status();
        if status == http::StatusCode::MOVED_PERMANENTLY {
            return Err(ApiError::moved_permanently(
                rsp.headers().get(http::header::LOCATION),
            ));
        } else if status == http::StatusCode::TOO_MANY_REQUESTS {
            return Err(ApiError::from_gitlab_rate_limit(rsp.headers()));
        }
        let v = if let Ok(v) = serde_json::from_slice(rsp.body()) {
            v
        } else {
            return Err(ApiError::server_error(status, rsp.body()));
        };
        if !status.is_success() {
            return Err(ApiError::from_gitlab_with_status(status, v));
        }

        serde_json::from_value::<T>(v).map_err(ApiError::data_type::<T>)
    }
}

#[cfg(test)]
mod tests {
    use std::time::Duration;

    use chrono::{TimeZone, Utc};
    use http::{header, HeaderMap, HeaderName, HeaderValue, StatusCode};
    use serde::Deserialize;
    use serde_json::json;
    use url::Url;

    use crate::api::endpoint_prelude::*;
    use crate::api::{ApiError, AsyncQuery, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    struct Dummy;

    impl Endpoint for Dummy {
        fn method(&self) -> Method {
            Method::GET
        }

        fn endpoint(&self) -> Cow<'static, str> {
            "dummy".into()
        }
    }

    struct DummyWithParameters;

    impl Endpoint for DummyWithParameters {
        fn method(&self) -> Method {
            Method::GET
        }

        fn endpoint(&self) -> Cow<'static, str> {
            "dummy/params".into()
        }

        fn parameters(&self) -> QueryParams<'_> {
            let mut params = QueryParams::default();
            params.push("dummy", true);
            params
        }

        fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
            let mut params = FormParams::default();
            params.push("dummy", true);
            params.into_body()
        }
    }

    #[test]
    fn test_endpoint_ref_parameters() {
        let endpoint = DummyWithParameters;
        let endpoint_ref = &endpoint;

        let params = endpoint_ref.parameters();
        let mut url = Url::parse("https://example.com").unwrap();
        params.add_to_url(&mut url);
        itertools::assert_equal(url.query_pairs(), [("dummy".into(), "true".into())]);
    }

    #[test]
    fn test_endpoint_ref_body() {
        let endpoint = DummyWithParameters;
        let endpoint_ref = &endpoint;

        let params = endpoint_ref.body().unwrap();
        if let Some((mime, bytes)) = params {
            assert_eq!(mime, "application/x-www-form-urlencoded");
            assert_eq!(bytes, b"dummy=true");
        } else {
            panic!("expected to have `dummy=true`");
        }
    }

    #[test]
    fn test_gitlab_redirect() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::MOVED_PERMANENTLY)
            .build()
            .unwrap();
        const OTHER_SITE: &str = "https://other.site/over/here";
        let headers = [(header::LOCATION, HeaderValue::from_static(OTHER_SITE))]
            .into_iter()
            .collect::<HeaderMap>();
        let client = SingleTestClient::new_raw_headers(endpoint, headers, "");

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::MovedPermanently {
            location, ..
        } = err
        {
            assert_eq!(location.as_deref(), Some(OTHER_SITE));
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[tokio::test]
    async fn test_gitlab_redirect_async() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::MOVED_PERMANENTLY)
            .build()
            .unwrap();
        const OTHER_SITE: &str = "https://other.site/over/here";
        let headers = [(header::LOCATION, HeaderValue::from_static(OTHER_SITE))]
            .into_iter()
            .collect::<HeaderMap>();
        let client = SingleTestClient::new_raw_headers(endpoint, headers, "");

        let res: Result<DummyResult, _> = Dummy.query_async(&client).await;
        let err = res.unwrap_err();
        if let ApiError::MovedPermanently {
            location, ..
        } = err
        {
            assert_eq!(location.as_deref(), Some(OTHER_SITE));
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_rate_limited() {
        let reset = Utc.with_ymd_and_hms(2024, 12, 31, 0, 0, 0).unwrap();
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::TOO_MANY_REQUESTS)
            .build()
            .unwrap();
        let headers = [
            ("ratelimit-limit", "5"),
            ("ratelimit-name", "gitlab_error_test"),
            ("ratelimit-observed", "100"),
            ("ratelimit-remaining", "10"),
            ("ratelimit-reset", "1735603200"),
            ("retry-after", "1000"),
        ]
        .into_iter()
        .map(|(n, v)| (HeaderName::from_static(n), HeaderValue::from_static(v)))
        .collect();
        let client = SingleTestClient::new_raw_headers(endpoint, headers, "");

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabRateLimited {
            rl_limit,
            rl_name,
            rl_observed,
            rl_remaining,
            rl_reset,
            retry_after,
        } = err
        {
            assert_eq!(rl_limit, 5);
            assert_eq!(rl_name, "gitlab_error_test");
            assert_eq!(rl_observed, 100);
            assert_eq!(rl_remaining, 10);
            assert_eq!(rl_reset, reset);
            assert_eq!(retry_after, Duration::from_secs(1000));
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[tokio::test]
    async fn test_gitlab_rate_limited_async() {
        let reset = Utc.with_ymd_and_hms(2024, 12, 31, 0, 0, 0).unwrap();
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::TOO_MANY_REQUESTS)
            .build()
            .unwrap();
        let headers = [
            ("ratelimit-limit", "5"),
            ("ratelimit-name", "gitlab_error_test"),
            ("ratelimit-observed", "100"),
            ("ratelimit-remaining", "10"),
            ("ratelimit-reset", "1735603200"),
            ("retry-after", "1000"),
        ]
        .into_iter()
        .map(|(n, v)| (HeaderName::from_static(n), HeaderValue::from_static(v)))
        .collect();
        let client = SingleTestClient::new_raw_headers(endpoint, headers, "");

        let res: Result<DummyResult, _> = Dummy.query_async(&client).await;
        let err = res.unwrap_err();
        if let ApiError::GitlabRateLimited {
            rl_limit,
            rl_name,
            rl_observed,
            rl_remaining,
            rl_reset,
            retry_after,
        } = err
        {
            assert_eq!(rl_limit, 5);
            assert_eq!(rl_name, "gitlab_error_test");
            assert_eq!(rl_observed, 100);
            assert_eq!(rl_remaining, 10);
            assert_eq!(rl_reset, reset);
            assert_eq!(retry_after, Duration::from_secs(1000));
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[derive(Debug, Deserialize)]
    struct DummyResult {
        value: u8,
    }

    #[test]
    fn test_gitlab_non_json_response() {
        let endpoint = ExpectedUrl::builder().endpoint("dummy").build().unwrap();
        let client = SingleTestClient::new_raw(endpoint, "not json");

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabService {
            status, ..
        } = err
        {
            assert_eq!(status, StatusCode::OK);
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_empty_response() {
        let endpoint = ExpectedUrl::builder().endpoint("dummy").build().unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabService {
            status, ..
        } = err
        {
            assert_eq!(status, StatusCode::OK);
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_error_bad_json() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::NOT_FOUND)
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabService {
            status, ..
        } = err
        {
            assert_eq!(status, StatusCode::NOT_FOUND);
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_error_detection() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::NOT_FOUND)
            .build()
            .unwrap();
        let client = SingleTestClient::new_json(
            endpoint,
            &json!({
                "message": "dummy error message",
            }),
        );

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabWithStatus {
            status,
            msg,
        } = err
        {
            assert_eq!(status, StatusCode::NOT_FOUND);
            assert_eq!(msg, "dummy error message");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_error_detection_legacy() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::NOT_FOUND)
            .build()
            .unwrap();
        let client = SingleTestClient::new_json(
            endpoint,
            &json!({
                "error": "dummy error message",
            }),
        );

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabWithStatus {
            status,
            msg,
        } = err
        {
            assert_eq!(status, StatusCode::NOT_FOUND);
            assert_eq!(msg, "dummy error message");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_gitlab_error_detection_unknown() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("dummy")
            .status(StatusCode::NOT_FOUND)
            .build()
            .unwrap();
        let err_obj = json!({
            "bogus": "dummy error message",
        });
        let client = SingleTestClient::new_json(endpoint, &err_obj);

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::GitlabUnrecognizedWithStatus {
            status,
            obj,
        } = err
        {
            assert_eq!(status, StatusCode::NOT_FOUND);
            assert_eq!(obj, err_obj);
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_bad_deserialization() {
        let endpoint = ExpectedUrl::builder().endpoint("dummy").build().unwrap();
        let client = SingleTestClient::new_json(
            endpoint,
            &json!({
                "not_value": 0,
            }),
        );

        let res: Result<DummyResult, _> = Dummy.query(&client);
        let err = res.unwrap_err();
        if let ApiError::DataType {
            source,
            typename,
        } = err
        {
            assert_eq!(source.to_string(), "missing field `value`");
            assert_eq!(typename, "gitlab::api::endpoint::tests::DummyResult");
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_good_deserialization() {
        let endpoint = ExpectedUrl::builder().endpoint("dummy").build().unwrap();
        let client = SingleTestClient::new_json(
            endpoint,
            &json!({
                "value": 0,
            }),
        );

        let res: DummyResult = Dummy.query(&client).unwrap();
        assert_eq!(res.value, 0);
    }

    #[tokio::test]
    async fn test_good_deserialization_async() {
        let endpoint = ExpectedUrl::builder().endpoint("dummy").build().unwrap();
        let client = SingleTestClient::new_json(
            endpoint,
            &json!({
                "value": 0,
            }),
        );

        let res: DummyResult = Dummy.query_async(&client).await.unwrap();
        assert_eq!(res.value, 0);
    }
}
