// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// List the users who starred the specified project.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct Starrers<'a> {
    /// The project to list the starrers for.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// Search for specific users.
    /// The search term is a simple contains check, to expressions.
    /// Matches on multiple fields simultaneously like name and username and returns all matching
    /// starrers.
    #[builder(setter(into), default)]
    search: Option<Cow<'a, str>>,
}

impl<'a> Starrers<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> StarrersBuilder<'a> {
        StarrersBuilder::default()
    }
}

impl<'a> Endpoint for Starrers<'a> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/starrers", self.project).into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params.push_opt("search", self.search.as_ref());

        params
    }
}

impl<'a> Pageable for Starrers<'a> {}

#[cfg(test)]
mod tests {
    use crate::api::projects::star::{Starrers, StarrersBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_necessary() {
        let err = Starrers::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, StarrersBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        Starrers::builder().project("project").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/project%2Fsubproject/starrers")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = Starrers::builder()
            .project("project/subproject")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_search() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/starrers")
            .add_query_params(&[("search", "query")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = Starrers::builder()
            .project("simple/project")
            .search("query")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
