// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Stars a given project.
#[derive(Debug, Builder, Clone)]
pub struct StarProject<'a> {
    /// The project to star.
    #[builder(setter(into))]
    project: NameOrId<'a>,
}

impl<'a> StarProject<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> StarProjectBuilder<'a> {
        StarProjectBuilder::default()
    }
}

impl<'a> Endpoint for StarProject<'a> {
    fn method(&self) -> Method {
        Method::POST
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/star", self.project).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::star::{StarProject, StarProjectBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_necessary() {
        let err = StarProject::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, StarProjectBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        StarProject::builder().project("project").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("projects/project%2Fsubproject/star")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = StarProject::builder()
            .project("project/subproject")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
