// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::{self, NameOrId, SortOrder};
use crate::api::endpoint_prelude::*;
use crate::api::ParamValue;

/// Sort order for commit statuses.
#[non_exhaustive]
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
pub enum CommitStatusesOrderBy {
    /// Order by the ID of the status.
    Id,
    /// Order by the ID of the pipeline.
    PipelineId,
}

impl CommitStatusesOrderBy {
    fn as_str(self) -> &'static str {
        match self {
            CommitStatusesOrderBy::Id => "id",
            CommitStatusesOrderBy::PipelineId => "pipeline_id",
        }
    }
}

impl ParamValue<'static> for CommitStatusesOrderBy {
    fn as_value(&self) -> Cow<'static, str> {
        self.as_str().into()
    }
}

/// Query for statuses on a specific commit in a project.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct CommitStatuses<'a> {
    /// The project to get a commit from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The commit to get statuses from.
    #[builder(setter(into))]
    commit: Cow<'a, str>,

    /// The branch or tag to check.
    #[builder(setter(into), default)]
    ref_: Option<Cow<'a, str>>,
    /// Filter by build stage.
    #[builder(setter(into), default)]
    stage: Option<Cow<'a, str>>,
    /// Filter by job name.
    #[builder(setter(into), default)]
    name: Option<Cow<'a, str>>,
    /// Return all statuses, not just the latest.
    #[builder(default)]
    all: Option<bool>,
    /// Filter by pipeline ID.
    #[builder(default)]
    pipeline_id: Option<u64>,
    /// Order results by a given key.
    #[builder(default)]
    order_by: Option<CommitStatusesOrderBy>,
    /// The sort order for the results.
    #[builder(default)]
    sort: Option<SortOrder>,
}

impl<'a> CommitStatuses<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> CommitStatusesBuilder<'a> {
        CommitStatusesBuilder::default()
    }
}

impl Endpoint for CommitStatuses<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/repository/commits/{}/statuses",
            self.project,
            common::path_escaped(&self.commit),
        )
        .into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params
            .push_opt("ref", self.ref_.as_ref())
            .push_opt("stage", self.stage.as_ref())
            .push_opt("name", self.name.as_ref())
            .push_opt("all", self.all)
            .push_opt("pipeline_id", self.pipeline_id)
            .push_opt("order_by", self.order_by)
            .push_opt("sort", self.sort.map(|value| value.as_str()));

        params
    }
}

impl Pageable for CommitStatuses<'_> {}

#[cfg(test)]
mod tests {
    use crate::api::common::SortOrder;
    use crate::api::projects::repository::commits::{
        CommitStatuses, CommitStatusesBuilderError, CommitStatusesOrderBy,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn test_commit_statuses_order_by_as_str() {
        let items = &[
            (CommitStatusesOrderBy::Id, "id"),
            (CommitStatusesOrderBy::PipelineId, "pipeline_id"),
        ];

        for (i, s) in items {
            assert_eq!(i.as_str(), *s);
        }
    }

    #[test]
    fn project_and_commit_are_necessary() {
        let err = CommitStatuses::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, CommitStatusesBuilderError, "project");
    }

    #[test]
    fn project_is_necessary() {
        let err = CommitStatuses::builder()
            .commit("master")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, CommitStatusesBuilderError, "project");
    }

    #[test]
    fn commit_is_necessary() {
        let err = CommitStatuses::builder().project(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, CommitStatusesBuilderError, "commit");
    }

    #[test]
    fn project_and_commit_are_sufficient() {
        CommitStatuses::builder()
            .project(1)
            .commit("master")
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_ref() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("ref", "master")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .ref_("master")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_stage() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("stage", "stage")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .stage("stage")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_name() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("name", "jobname")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .name("jobname")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_all() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("all", "false")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .all(false)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_pipeline_id() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("pipeline_id", "123")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .pipeline_id(123)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_order_by() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("order_by", "pipeline_id")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .order_by(CommitStatusesOrderBy::PipelineId)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_sort() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/repository/commits/0000000000000000000000000000000000000000/statuses")
            .add_query_params(&[("sort", "desc")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = CommitStatuses::builder()
            .project("simple/project")
            .commit("0000000000000000000000000000000000000000")
            .sort(SortOrder::Descending)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
