// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;
use crate::api::projects::{ContainerExpirationKeepN, ContainerExpirationOlderThan};

/// Delete registry repository tags based on given criteria.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct DeleteRepositoryTags<'a> {
    /// The project to delete from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The repository id to delete from.
    repository_id: u64,
    /// The re2 regex of the tags to delete. To delete all tags specify ".*".
    #[builder(setter(into), default)]
    name_regex_delete: Option<Cow<'a, str>>,
    /// The re2 regex of the tags to keep.
    ///
    /// This value overrides any matches from `name_regex_delete`.
    #[builder(setter(into), default)]
    name_regex_keep: Option<Cow<'a, str>>,
    /// The amount of latest tags to keep.
    ///
    /// This value takes precedence over `name_regex_delete`, so no matter what, this number of
    /// tags will be kept.
    #[builder(setter(into), default)]
    keep_n: Option<ContainerExpirationKeepN>,
    /// Tags to delete that are older than the given time.
    #[builder(setter(into), default)]
    older_than: Option<ContainerExpirationOlderThan>,
}

impl<'a> DeleteRepositoryTags<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DeleteRepositoryTagsBuilder<'a> {
        DeleteRepositoryTagsBuilder::default()
    }
}

impl Endpoint for DeleteRepositoryTags<'_> {
    fn method(&self) -> Method {
        Method::DELETE
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/registry/repositories/{}/tags",
            self.project, self.repository_id,
        )
        .into()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        let mut params = FormParams::default();

        params
            .push_opt("name_regex_delete", self.name_regex_delete.as_ref())
            .push_opt("name_regex_keep", self.name_regex_keep.as_ref())
            .push_opt("keep_n", self.keep_n)
            .push_opt("older_than", self.older_than);

        params.into_body()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::registry::{DeleteRepositoryTags, DeleteRepositoryTagsBuilderError};
    use crate::api::projects::{ContainerExpirationKeepN, ContainerExpirationOlderThan};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn parameters_are_necessary() {
        let err = DeleteRepositoryTags::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, DeleteRepositoryTagsBuilderError, "project");
    }

    #[test]
    fn project_is_necessary() {
        let err = DeleteRepositoryTags::builder()
            .repository_id(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DeleteRepositoryTagsBuilderError, "project");
    }

    #[test]
    fn repository_is_necessary() {
        let err = DeleteRepositoryTags::builder()
            .project(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DeleteRepositoryTagsBuilderError, "repository_id");
    }

    #[test]
    fn all_parameters_are_sufficient() {
        DeleteRepositoryTags::builder()
            .project(1)
            .repository_id(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/simple%2Fproject/registry/repositories/1/tags")
            .content_type("application/x-www-form-urlencoded")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteRepositoryTags::builder()
            .project("simple/project")
            .repository_id(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_name_regex_delete() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/simple%2Fproject/registry/repositories/1/tags")
            .content_type("application/x-www-form-urlencoded")
            .body_str("name_regex_delete=some-other-regex")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteRepositoryTags::builder()
            .project("simple/project")
            .repository_id(1)
            .name_regex_delete("some-other-regex")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_name_regex_keep() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/simple%2Fproject/registry/repositories/1/tags")
            .content_type("application/x-www-form-urlencoded")
            .body_str("name_regex_keep=some-regex")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteRepositoryTags::builder()
            .project("simple/project")
            .repository_id(1)
            .name_regex_keep("some-regex")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_keep_n() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/simple%2Fproject/registry/repositories/1/tags")
            .content_type("application/x-www-form-urlencoded")
            .body_str("keep_n=50")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteRepositoryTags::builder()
            .project("simple/project")
            .repository_id(1)
            .keep_n(ContainerExpirationKeepN::Fifty)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_older_than() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/simple%2Fproject/registry/repositories/1/tags")
            .content_type("application/x-www-form-urlencoded")
            .body_str("older_than=7d")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteRepositoryTags::builder()
            .project("simple/project")
            .repository_id(1)
            .older_than(ContainerExpirationOlderThan::OneWeek)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
