// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::collections::BTreeSet;

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;
use crate::api::projects::jobs::JobScope;

/// Query for bridges on a pipeline.
#[derive(Debug, Builder, Clone)]
pub struct PipelineBridges<'a> {
    /// The project to query for the pipeline.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The ID of the pipeline.
    pipeline: u64,

    /// The scopes to filter jobs by.
    #[builder(setter(name = "_scopes"), default, private)]
    scopes: BTreeSet<JobScope>,
}

impl<'a> PipelineBridges<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> PipelineBridgesBuilder<'a> {
        PipelineBridgesBuilder::default()
    }
}

impl PipelineBridgesBuilder<'_> {
    /// Filter jobs by a scope.
    pub fn scope(&mut self, scope: JobScope) -> &mut Self {
        self.scopes.get_or_insert_with(BTreeSet::new).insert(scope);
        self
    }

    /// Filter jobs by a set of scopes.
    pub fn scopes<I>(&mut self, scopes: I) -> &mut Self
    where
        I: Iterator<Item = JobScope>,
    {
        self.scopes.get_or_insert_with(BTreeSet::new).extend(scopes);
        self
    }
}

impl Endpoint for PipelineBridges<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/pipelines/{}/bridges",
            self.project, self.pipeline,
        )
        .into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params.extend(self.scopes.iter().map(|&value| ("scope[]", value)));

        params
    }
}

impl Pageable for PipelineBridges<'_> {}

#[cfg(test)]
mod tests {
    use crate::api::projects::jobs::JobScope;
    use crate::api::projects::pipelines::{PipelineBridges, PipelineBridgesBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_and_pipeline_are_needed() {
        let err = PipelineBridges::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, PipelineBridgesBuilderError, "project");
    }

    #[test]
    fn project_is_needed() {
        let err = PipelineBridges::builder().pipeline(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, PipelineBridgesBuilderError, "project");
    }

    #[test]
    fn pipeline_is_needed() {
        let err = PipelineBridges::builder().project(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, PipelineBridgesBuilderError, "pipeline");
    }

    #[test]
    fn project_and_pipeline_are_sufficient() {
        PipelineBridges::builder()
            .project(1)
            .pipeline(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/simple%2Fproject/pipelines/1/bridges")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = PipelineBridges::builder()
            .project("simple/project")
            .pipeline(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_scopes() {
        let endpoint = ExpectedUrl::builder()
            .endpoint("projects/1/pipelines/1/bridges")
            .add_query_params(&[("scope[]", "created"), ("scope[]", "success")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = PipelineBridges::builder()
            .project(1)
            .pipeline(1)
            .scope(JobScope::Created)
            .scopes([JobScope::Created, JobScope::Success].iter().cloned())
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
