// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// List Pages settings for the project.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct EditPages<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,

    /// If unique domain is enabled.
    #[builder(default)]
    pages_unique_domain_enabled: Option<bool>,
    /// true if the project is set to force HTTPS.
    #[builder(default)]
    pages_https_only: Option<bool>,
    /// The primary domain to use for pages deployments.
    #[builder(setter(into), default)]
    pages_primary_domain: Option<Cow<'a, str>>,
}

impl<'a> EditPages<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> EditPagesBuilder<'a> {
        EditPagesBuilder::default()
    }
}

impl Endpoint for EditPages<'_> {
    fn method(&self) -> Method {
        Method::PATCH
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/pages", self.project).into()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        let mut params = FormParams::default();

        params
            .push_opt(
                "pages_unique_domain_enabled",
                self.pages_unique_domain_enabled,
            )
            .push_opt("pages_https_only", self.pages_https_only)
            .push_opt("pages_primary_domain", self.pages_primary_domain.as_ref());

        params.into_body()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::pages::{EditPages, EditPagesBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = EditPages::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, EditPagesBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        EditPages::builder().project("blah").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::PATCH)
            .endpoint("projects/blah/pages")
            .content_type("application/x-www-form-urlencoded")
            .body_str("")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = EditPages::builder().project("blah").build().unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_pages_unique_domain_enabled() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::PATCH)
            .endpoint("projects/blah/pages")
            .content_type("application/x-www-form-urlencoded")
            .body_str("pages_unique_domain_enabled=true")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = EditPages::builder()
            .project("blah")
            .pages_unique_domain_enabled(true)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_pages_https_only() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::PATCH)
            .endpoint("projects/blah/pages")
            .content_type("application/x-www-form-urlencoded")
            .body_str("pages_https_only=true")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = EditPages::builder()
            .project("blah")
            .pages_https_only(true)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_pages_primary_domain() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::PATCH)
            .endpoint("projects/blah/pages")
            .content_type("application/x-www-form-urlencoded")
            .body_str("pages_primary_domain=alternative.domain.internal")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = EditPages::builder()
            .project("blah")
            .pages_primary_domain("alternative.domain.internal")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
