// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::{self, NameOrId};
use crate::api::endpoint_prelude::*;

/// Download job artifacts.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct DownloadJobArtifactFile<'a> {
    /// The project to download artifacts from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The job to download artifacts from.
    job: u64,
    /// The path to the file to download from the artifacts.
    #[builder(setter(into))]
    path: Cow<'a, str>,
    /// The job token to authenticate with.
    ///
    /// For use with multi-project pipelines.
    #[builder(setter(into), default)]
    job_token: Option<Cow<'a, str>>,
}

impl<'a> DownloadJobArtifactFile<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DownloadJobArtifactFileBuilder<'a> {
        DownloadJobArtifactFileBuilder::default()
    }
}

impl Endpoint for DownloadJobArtifactFile<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/jobs/{}/artifacts/{}",
            self.project,
            self.job,
            common::directory_path_escaped(&self.path),
        )
        .into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params.push_opt("job_token", self.job_token.as_ref());

        params
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::jobs::artifacts::{
        DownloadJobArtifactFile, DownloadJobArtifactFileBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DownloadJobArtifactFile::builder()
            .job(1)
            .path("file")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadJobArtifactFileBuilderError, "project");
    }

    #[test]
    fn job_is_required() {
        let err = DownloadJobArtifactFile::builder()
            .project(1)
            .path("file")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadJobArtifactFileBuilderError, "job");
    }

    #[test]
    fn path_is_required() {
        let err = DownloadJobArtifactFile::builder()
            .project(1)
            .job(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadJobArtifactFileBuilderError, "path");
    }

    #[test]
    fn sufficient_parameters() {
        DownloadJobArtifactFile::builder()
            .project(1)
            .job(1)
            .path("file")
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts/path/to/file")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadJobArtifactFile::builder()
            .project("example/repo")
            .job(1)
            .path("path/to/file")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_job_token() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts/path/to/file")
            .add_query_params(&[("job_token", "TOKEN")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadJobArtifactFile::builder()
            .project("example/repo")
            .job(1)
            .path("path/to/file")
            .job_token("TOKEN")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
