// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::{self, NameOrId};
use crate::api::endpoint_prelude::*;

/// Download job artifacts.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct DownloadRefJobArtifacts<'a> {
    /// The project to download artifacts from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The name of the job to download artifacts from.
    #[builder(setter(into))]
    job: Cow<'a, str>,
    /// The ref to download artifacts from.
    #[builder(setter(into))]
    ref_name: Cow<'a, str>,
    /// The job token to authenticate with.
    ///
    /// For use with multi-project pipelines.
    #[builder(setter(into), default)]
    job_token: Option<Cow<'a, str>>,
}

impl<'a> DownloadRefJobArtifacts<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DownloadRefJobArtifactsBuilder<'a> {
        DownloadRefJobArtifactsBuilder::default()
    }
}

impl DownloadRefJobArtifactsBuilder<'_> {
    /// Download an artifact from a merge request.
    pub fn ref_name_merge_request(&mut self, merge_request: u64) -> &mut Self {
        self.ref_name = Some(format!("refs/merge-requests/{merge_request}/head").into());
        self
    }
}

impl Endpoint for DownloadRefJobArtifacts<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/jobs/artifacts/{}/download",
            self.project,
            common::path_escaped(&self.ref_name),
        )
        .into()
    }

    fn parameters(&self) -> QueryParams<'_> {
        let mut params = QueryParams::default();

        params
            .push("job", &self.job)
            .push_opt("job_token", self.job_token.as_ref());

        params
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::jobs::artifacts::{
        DownloadRefJobArtifacts, DownloadRefJobArtifactsBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DownloadRefJobArtifacts::builder()
            .job("job")
            .ref_name("master")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadRefJobArtifactsBuilderError, "project");
    }

    #[test]
    fn job_is_required() {
        let err = DownloadRefJobArtifacts::builder()
            .project(1)
            .ref_name("master")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadRefJobArtifactsBuilderError, "job");
    }

    #[test]
    fn ref_name_is_required() {
        let err = DownloadRefJobArtifacts::builder()
            .project(1)
            .job("job")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DownloadRefJobArtifactsBuilderError, "ref_name");
    }

    #[test]
    fn sufficient_parameters() {
        DownloadRefJobArtifacts::builder()
            .project(1)
            .job("job")
            .ref_name("master")
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/artifacts/master/download")
            .add_query_params(&[("job", "job")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadRefJobArtifacts::builder()
            .project("example/repo")
            .job("job")
            .ref_name("master")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_via_merge_request() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint(
                "projects/example%2Frepo/jobs/artifacts/refs%2Fmerge-requests%2F1%2Fhead/download",
            )
            .add_query_params(&[("job", "job")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadRefJobArtifacts::builder()
            .project("example/repo")
            .job("job")
            .ref_name_merge_request(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_job_token() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/example%2Frepo/jobs/artifacts/master/download")
            .add_query_params(&[("job", "job"), ("job_token", "TOKEN")])
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DownloadRefJobArtifacts::builder()
            .project("example/repo")
            .job("job")
            .ref_name("master")
            .job_token("TOKEN")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
