// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Delete job artifacts.
#[derive(Debug, Builder, Clone)]
pub struct DeleteJobArtifacts<'a> {
    /// The project to delete artifacts from.
    #[builder(setter(into))]
    project: NameOrId<'a>,
    /// The job to delete artifacts from.
    job: u64,
}

impl<'a> DeleteJobArtifacts<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DeleteJobArtifactsBuilder<'a> {
        DeleteJobArtifactsBuilder::default()
    }
}

impl Endpoint for DeleteJobArtifacts<'_> {
    fn method(&self) -> Method {
        Method::DELETE
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/jobs/{}/artifacts", self.project, self.job).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::jobs::artifacts::{
        DeleteJobArtifacts, DeleteJobArtifactsBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DeleteJobArtifacts::builder().job(1).build().unwrap_err();
        crate::test::assert_missing_field!(err, DeleteJobArtifactsBuilderError, "project");
    }

    #[test]
    fn job_is_required() {
        let err = DeleteJobArtifacts::builder()
            .project(1)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, DeleteJobArtifactsBuilderError, "job");
    }

    #[test]
    fn project_and_job_are_sufficient() {
        DeleteJobArtifacts::builder()
            .project(1)
            .job(1)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/example%2Frepo/jobs/1/artifacts")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DeleteJobArtifacts::builder()
            .project("example/repo")
            .job(1)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
