// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Fetch the 'CI/CD job token access settings' (job token scope) of a project.
#[derive(Debug, Builder, Clone)]
pub struct JobTokenScopes<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,
}

impl<'a> JobTokenScopes<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> JobTokenScopesBuilder<'a> {
        JobTokenScopesBuilder::default()
    }
}

impl Endpoint for JobTokenScopes<'_> {
    fn method(&self) -> Method {
        Method::GET
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/job_token_scope", self.project).into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::job_token_scopes::{JobTokenScopes, JobTokenScopesBuilderError};
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = JobTokenScopes::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, JobTokenScopesBuilderError, "project");
    }

    #[test]
    fn project_is_sufficient() {
        JobTokenScopes::builder().project("blah").build().unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::GET)
            .endpoint("projects/blah/job_token_scope")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = JobTokenScopes::builder().project("blah").build().unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
