// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Edit the 'Limit access to this project setting' (job token scope) of a project.
#[derive(Debug, Builder, Clone)]
pub struct EditJobTokenScope<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,

    /// Indicates if the 'Limit access to this project' setting should be enabled.
    enabled: bool,
}

impl<'a> EditJobTokenScope<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> EditJobTokenScopeBuilder<'a> {
        EditJobTokenScopeBuilder::default()
    }
}

impl Endpoint for EditJobTokenScope<'_> {
    fn method(&self) -> Method {
        Method::PATCH
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("projects/{}/job_token_scope", self.project).into()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        let mut params = FormParams::default();

        params.push("enabled", self.enabled);

        params.into_body()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::job_token_scopes::{
        EditJobTokenScope, EditJobTokenScopeBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = EditJobTokenScope::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, EditJobTokenScopeBuilderError, "project");
    }

    #[test]
    fn enabled_is_required() {
        let err = EditJobTokenScope::builder()
            .project("blah")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, EditJobTokenScopeBuilderError, "enabled");
    }

    #[test]
    fn project_and_enabled_is_sufficient() {
        EditJobTokenScope::builder()
            .project("blah")
            .enabled(true)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::PATCH)
            .endpoint("projects/blah/job_token_scope")
            .content_type("application/x-www-form-urlencoded")
            .body_str("enabled=false")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = EditJobTokenScope::builder()
            .project("blah")
            .enabled(false)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
