// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use derive_builder::Builder;

use crate::api::common::NameOrId;
use crate::api::endpoint_prelude::*;

/// Remove a project from the 'CI/CD job token inbound allowlist' of a project.
#[derive(Debug, Builder, Clone)]
pub struct DisallowJobTokenProject<'a> {
    /// The ID or URL-encoded path of the project.
    #[builder(setter(into))]
    project: NameOrId<'a>,

    /// The ID of the project that is removed from the 'CI/CD job token inbound allowlist'.
    target_project_id: u64,
}

impl<'a> DisallowJobTokenProject<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> DisallowJobTokenProjectBuilder<'a> {
        DisallowJobTokenProjectBuilder::default()
    }
}

impl Endpoint for DisallowJobTokenProject<'_> {
    fn method(&self) -> Method {
        Method::DELETE
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!(
            "projects/{}/job_token_scope/allowlist/{}",
            self.project, self.target_project_id
        )
        .into()
    }
}

#[cfg(test)]
mod tests {
    use http::Method;

    use crate::api::projects::job_token_scopes::{
        DisallowJobTokenProject, DisallowJobTokenProjectBuilderError,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn project_is_required() {
        let err = DisallowJobTokenProject::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, DisallowJobTokenProjectBuilderError, "project");
    }

    #[test]
    fn target_is_required() {
        let err = DisallowJobTokenProject::builder()
            .project("blah")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(
            err,
            DisallowJobTokenProjectBuilderError,
            "target_project_id",
        );
    }

    #[test]
    fn project_and_target_is_sufficient() {
        DisallowJobTokenProject::builder()
            .project("blah")
            .target_project_id(42)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::DELETE)
            .endpoint("projects/blah/job_token_scope/allowlist/42")
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = DisallowJobTokenProject::builder()
            .project("blah")
            .target_project_id(42)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
