// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use chrono::NaiveDate;
use derive_builder::Builder;

use crate::api::common::{AccessLevel, NameOrId};
use crate::api::endpoint_prelude::*;
use crate::api::ParamValue;

/// Tasks users may be assigned upon addition to a group.
#[derive(Debug, Clone, Copy, PartialEq, Eq)]
#[non_exhaustive]
pub enum GroupInviteTasksToBeDone {
    /// Request to focus on CI tasks.
    Ci,
    /// Request to focus on coding tasks.
    Code,
    /// Request to focus on issues.
    Issues,
}

impl GroupInviteTasksToBeDone {
    /// The tasks as a query parameter.
    fn as_str(self) -> &'static str {
        match self {
            GroupInviteTasksToBeDone::Ci => "ci",
            GroupInviteTasksToBeDone::Code => "code",
            GroupInviteTasksToBeDone::Issues => "issues",
        }
    }
}

impl ParamValue<'static> for GroupInviteTasksToBeDone {
    fn as_value(&self) -> Cow<'static, str> {
        self.as_str().into()
    }
}

/// Add a user as a member of a group.
#[derive(Debug, Builder, Clone)]
#[builder(setter(strip_option))]
pub struct AddGroupMember<'a> {
    /// The group to add the user to.
    #[builder(setter(into))]
    group: NameOrId<'a>,
    /// The user to add to the group.
    #[builder(setter(into))]
    user: NameOrId<'a>,
    /// The access level for the user in the group.
    access_level: AccessLevel,

    /// When the user's access expires.
    #[builder(default)]
    expires_at: Option<NaiveDate>,
    /// The source of the invitation.
    #[builder(setter(into), default)]
    invite_source: Option<Cow<'a, str>>,
    /// ID of a custom role to assign to the member.
    #[builder(default)]
    member_role_id: Option<u64>,
}

impl<'a> AddGroupMember<'a> {
    /// Create a builder for the endpoint.
    pub fn builder() -> AddGroupMemberBuilder<'a> {
        AddGroupMemberBuilder::default()
    }
}

impl Endpoint for AddGroupMember<'_> {
    fn method(&self) -> Method {
        Method::POST
    }

    fn endpoint(&self) -> Cow<'static, str> {
        format!("groups/{}/members", self.group).into()
    }

    fn body(&self) -> Result<Option<(&'static str, Vec<u8>)>, BodyError> {
        let mut params = FormParams::default();

        params
            .push("access_level", self.access_level.as_u64())
            .push_opt("expires_at", self.expires_at)
            .push_opt("invite_source", self.invite_source.as_ref())
            .push_opt("member_role_id", self.member_role_id);

        match &self.user {
            NameOrId::Name(name) => params.push("username", name),
            NameOrId::Id(id) => params.push("user_id", *id),
        };

        params.into_body()
    }
}

#[cfg(test)]
mod tests {
    use chrono::NaiveDate;
    use http::Method;

    use crate::api::common::AccessLevel;
    use crate::api::groups::members::{
        AddGroupMember, AddGroupMemberBuilderError, GroupInviteTasksToBeDone,
    };
    use crate::api::{self, Query};
    use crate::test::client::{ExpectedUrl, SingleTestClient};

    #[test]
    fn group_invite_tasks_as_str() {
        let items = &[
            (GroupInviteTasksToBeDone::Ci, "ci"),
            (GroupInviteTasksToBeDone::Code, "code"),
            (GroupInviteTasksToBeDone::Issues, "issues"),
        ];

        for (i, s) in items {
            assert_eq!(i.as_str(), *s);
        }
    }

    #[test]
    fn all_parameters_are_needed() {
        let err = AddGroupMember::builder().build().unwrap_err();
        crate::test::assert_missing_field!(err, AddGroupMemberBuilderError, "group");
    }

    #[test]
    fn group_is_necessary() {
        let err = AddGroupMember::builder()
            .user(1)
            .access_level(AccessLevel::Developer)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, AddGroupMemberBuilderError, "group");
    }

    #[test]
    fn user_is_necessary() {
        let err = AddGroupMember::builder()
            .group(1)
            .access_level(AccessLevel::Developer)
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, AddGroupMemberBuilderError, "user");
    }

    #[test]
    fn access_level_is_necessary() {
        let err = AddGroupMember::builder()
            .group(1)
            .user("name")
            .build()
            .unwrap_err();
        crate::test::assert_missing_field!(err, AddGroupMemberBuilderError, "access_level");
    }

    #[test]
    fn sufficient_parameters() {
        AddGroupMember::builder()
            .group("group")
            .user(1)
            .access_level(AccessLevel::Developer)
            .build()
            .unwrap();
    }

    #[test]
    fn endpoint() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("groups/group%2Fsubgroup/members")
            .content_type("application/x-www-form-urlencoded")
            .body_str(concat!("access_level=30", "&user_id=1"))
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AddGroupMember::builder()
            .group("group/subgroup")
            .user(1)
            .access_level(AccessLevel::Developer)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_username() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("groups/group%2Fsubgroup/members")
            .content_type("application/x-www-form-urlencoded")
            .body_str(concat!("access_level=30", "&username=uname"))
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AddGroupMember::builder()
            .group("group/subgroup")
            .user("uname")
            .access_level(AccessLevel::Developer)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_expires_at() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("groups/group%2Fsubgroup/members")
            .content_type("application/x-www-form-urlencoded")
            .body_str(concat!(
                "access_level=30",
                "&expires_at=2020-01-01",
                "&user_id=1",
            ))
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AddGroupMember::builder()
            .group("group/subgroup")
            .user(1)
            .access_level(AccessLevel::Developer)
            .expires_at(NaiveDate::from_ymd_opt(2020, 1, 1).unwrap())
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_invite_source() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("groups/group%2Fsubgroup/members")
            .content_type("application/x-www-form-urlencoded")
            .body_str(concat!(
                "access_level=30",
                "&invite_source=tuesday",
                "&user_id=1",
            ))
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AddGroupMember::builder()
            .group("group/subgroup")
            .user(1)
            .access_level(AccessLevel::Developer)
            .invite_source("tuesday")
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }

    #[test]
    fn endpoint_member_role_id() {
        let endpoint = ExpectedUrl::builder()
            .method(Method::POST)
            .endpoint("groups/group%2Fsubgroup/members")
            .content_type("application/x-www-form-urlencoded")
            .body_str(concat!(
                "access_level=30",
                "&member_role_id=14",
                "&user_id=1",
            ))
            .build()
            .unwrap();
        let client = SingleTestClient::new_raw(endpoint, "");

        let endpoint = AddGroupMember::builder()
            .group("group/subgroup")
            .user(1)
            .access_level(AccessLevel::Developer)
            .member_role_id(14)
            .build()
            .unwrap();
        api::ignore(endpoint).query(&client).unwrap();
    }
}
