// Copyright Kitware, Inc.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

#![warn(missing_docs)]

//! Basic Checks
//!
//! Simple checks useful in a wide variety of projects.

#[cfg(feature = "config")]
#[macro_use]
extern crate git_checks_config;

#[macro_use]
extern crate derive_builder;

#[macro_use]
extern crate lazy_static;

#[macro_use]
extern crate log;

#[cfg(feature = "config")]
#[macro_use]
extern crate serde_derive;

#[cfg(all(feature = "config", test))]
#[macro_use]
extern crate serde_json;

mod crates {
    // public
    pub extern crate git_checks_core;
    pub extern crate git_workarea;

    #[cfg(feature = "config")]
    pub extern crate git_checks_config;
    #[cfg(feature = "config")]
    pub extern crate inventory;
    #[cfg(feature = "config")]
    pub extern crate serde;

    // private
    // pub extern crate derive_builder;
    pub extern crate itertools;
    pub extern crate rayon;
    pub extern crate regex;
    pub extern crate thiserror;
    pub extern crate ttl_cache;
    pub extern crate wait_timeout;

    #[cfg(test)]
    pub extern crate serde_json;
    #[cfg(test)]
    pub extern crate tempdir;
}

mod binary_format;

mod allow_robot;
pub use self::allow_robot::AllowRobot;

mod bad_commit;
pub use self::bad_commit::BadCommit;

mod bad_commits;
pub use self::bad_commits::BadCommits;

mod changelog;
pub use self::changelog::Changelog;
pub use self::changelog::ChangelogStyle;

mod check_end_of_line;
pub use self::check_end_of_line::CheckEndOfLine;

mod check_executable_permissions;
pub use self::check_executable_permissions::CheckExecutablePermissions;

mod check_size;
pub use self::check_size::CheckSize;

mod check_whitespace;
pub use self::check_whitespace::CheckWhitespace;

mod commit_subject;
pub use self::commit_subject::CommitSubject;

mod formatting;
pub use self::formatting::Formatting;

mod invalid_paths;
pub use self::invalid_paths::InvalidPaths;

mod invalid_utf8;
pub use self::invalid_utf8::InvalidUtf8;

mod lfs_pointer;
pub use self::lfs_pointer::LfsPointer;

mod reject_binaries;
pub use self::reject_binaries::RejectBinaries;

mod reject_conflict_paths;
pub use self::reject_conflict_paths::RejectConflictPaths;

mod reject_merges;
pub use self::reject_merges::RejectMerges;

mod reject_separate_root;
pub use self::reject_separate_root::RejectSeparateRoot;

mod reject_symlinks;
pub use self::reject_symlinks::RejectSymlinks;

mod release_branch;
pub use self::release_branch::ReleaseBranch;

mod restricted_path;
pub use self::restricted_path::RestrictedPath;

mod submodule_available;
pub use self::submodule_available::SubmoduleAvailable;

mod submodule_rewind;
pub use self::submodule_rewind::SubmoduleRewind;

mod submodule_watch;
pub use self::submodule_watch::SubmoduleWatch;

mod third_party;
pub use self::third_party::ThirdParty;

mod valid_name;
pub use self::valid_name::ValidName;
pub use self::valid_name::ValidNameFullNamePolicy;

/// Module for all builders for checks.
///
/// This module is here for documentation purposes. Each check's `builder()` method should be
/// used instead.
pub mod builders {
    pub use allow_robot::AllowRobotBuilder;
    pub use bad_commit::BadCommitBuilder;
    pub use bad_commits::BadCommitsBuilder;
    pub use changelog::ChangelogBuilder;
    pub use check_end_of_line::CheckEndOfLineBuilder;
    pub use check_executable_permissions::CheckExecutablePermissionsBuilder;
    pub use check_size::CheckSizeBuilder;
    pub use check_whitespace::CheckWhitespaceBuilder;
    pub use commit_subject::CommitSubjectBuilder;
    pub use formatting::FormattingBuilder;
    pub use invalid_paths::InvalidPathsBuilder;
    pub use invalid_utf8::InvalidUtf8Builder;
    pub use lfs_pointer::LfsPointerBuilder;
    pub use reject_binaries::RejectBinariesBuilder;
    pub use reject_conflict_paths::RejectConflictPathsBuilder;
    pub use reject_merges::RejectMergesBuilder;
    pub use reject_separate_root::RejectSeparateRootBuilder;
    pub use reject_symlinks::RejectSymlinksBuilder;
    pub use release_branch::ReleaseBranchBuilder;
    pub use restricted_path::RestrictedPathBuilder;
    pub use submodule_available::SubmoduleAvailableBuilder;
    pub use submodule_rewind::SubmoduleRewindBuilder;
    pub use submodule_watch::SubmoduleWatchBuilder;
    pub use third_party::ThirdPartyBuilder;
    pub use valid_name::ValidNameBuilder;
}

/// Configuration structures for checks.
///
/// These structures are registered using `inventory` using the `git-checks-config` crate. They
/// are offered here for documentation purposes mainly, but also in case their specific
/// implementations are useful.
#[cfg(feature = "config")]
pub mod config {
    pub use allow_robot::config::AllowRobotConfig;
    pub use bad_commit::config::BadCommitConfig;
    pub use bad_commits::config::BadCommitsConfig;
    pub use changelog::config::ChangelogConfig;
    pub use check_end_of_line::config::CheckEndOfLineConfig;
    pub use check_executable_permissions::config::CheckExecutablePermissionsConfig;
    pub use check_size::config::CheckSizeConfig;
    pub use check_whitespace::config::CheckWhitespaceConfig;
    pub use commit_subject::config::CommitSubjectConfig;
    pub use formatting::config::FormattingConfig;
    pub use invalid_paths::config::InvalidPathsConfig;
    pub use invalid_utf8::config::InvalidUtf8Config;
    pub use lfs_pointer::config::LfsPointerConfig;
    pub use reject_binaries::config::RejectBinariesConfig;
    pub use reject_conflict_paths::config::RejectConflictPathsConfig;
    pub use reject_merges::config::RejectMergesConfig;
    pub use reject_separate_root::config::RejectSeparateRootConfig;
    pub use reject_symlinks::config::RejectSymlinksConfig;
    pub use release_branch::config::ReleaseBranchConfig;
    pub use restricted_path::config::RestrictedPathConfig;
    pub use submodule_available::config::SubmoduleAvailableConfig;
    pub use submodule_rewind::config::SubmoduleRewindConfig;
    pub use submodule_watch::config::SubmoduleWatchConfig;
    pub use third_party::config::ThirdPartyConfig;
    pub use valid_name::config::ValidNameConfig;
}

#[cfg(test)]
pub mod test;
