// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::env;
use std::fmt::Debug;
use std::iter;
use std::thread;
use std::time::Duration;

use graphql_client::{GraphQLQuery, QueryBody, Response};
use itertools::Itertools;
use log::{info, warn};
use reqwest::blocking::Client;
use reqwest::header::{self, HeaderMap, HeaderValue};
use reqwest::Url;
use serde::{Deserialize, Serialize};
use serde_json::Value;
use thiserror::Error;

use crate::authorization::{
    CurrentUser, GithubAuthError, GithubAuthorization, InstallationAction, InstallationBehavior,
};

// The maximum number of times we will retry server errors.
const BACKOFF_LIMIT: usize = if cfg!(test) { 2 } else { 5 };
// The number of seconds to start retries at.
const BACKOFF_START: Duration = Duration::from_secs(1);
// How much to scale retry timeouts for a single query.
const BACKOFF_SCALE: u32 = 2;

#[derive(Debug, Error)]
#[non_exhaustive]
pub enum GithubError {
    #[error("url parse error: {}", source)]
    UrlParse {
        #[from]
        source: url::ParseError,
    },
    #[error("invalid `GITHUB_TOKEN`: {}", source)]
    InvalidToken {
        #[source]
        source: env::VarError,
    },
    #[error("invalid `GITHUB_ACTOR`: {}", source)]
    InvalidActor {
        #[source]
        source: env::VarError,
    },
    #[error("failed to send request to {}: {}", endpoint, source)]
    SendRequest {
        endpoint: Url,
        #[source]
        source: reqwest::Error,
    },
    #[error("github error: {}", response)]
    Github { response: String },
    #[error("deserialize error: {}", source)]
    Deserialize {
        #[from]
        source: serde_json::Error,
    },
    #[error("github service error: {}", status)]
    GithubService { status: reqwest::StatusCode },
    #[error("json response deserialize: {}", source)]
    JsonResponse {
        #[source]
        source: reqwest::Error,
    },
    #[allow(clippy::upper_case_acronyms)]
    #[error("graphql error: [\"{}\"]", message.iter().format("\", \""))]
    GraphQL { message: Vec<graphql_client::Error> },
    #[error("no response from github")]
    NoResponse {},
    #[error("failure even after exponential backoff")]
    GithubBackoff {},
    #[error("authorization error: {}", source)]
    Authorization {
        #[from]
        source: GithubAuthError,
    },
}

impl GithubError {
    fn should_backoff(&self) -> bool {
        matches!(self, GithubError::GithubService { .. })
    }

    pub(crate) fn send_request(endpoint: Url, source: reqwest::Error) -> Self {
        GithubError::SendRequest {
            endpoint,
            source,
        }
    }

    pub(crate) fn github(response: String) -> Self {
        GithubError::Github {
            response,
        }
    }

    fn github_service(status: reqwest::StatusCode) -> Self {
        GithubError::GithubService {
            status,
        }
    }

    pub(crate) fn json_response(source: reqwest::Error) -> Self {
        GithubError::JsonResponse {
            source,
        }
    }

    pub(crate) fn invalid_token(source: env::VarError) -> Self {
        GithubError::InvalidToken {
            source,
        }
    }

    pub(crate) fn invalid_actor(source: env::VarError) -> Self {
        GithubError::InvalidActor {
            source,
        }
    }

    fn graphql(message: Vec<graphql_client::Error>) -> Self {
        GithubError::GraphQL {
            message,
        }
    }

    fn no_response() -> Self {
        GithubError::NoResponse {}
    }

    fn github_backoff() -> Self {
        GithubError::GithubBackoff {}
    }
}

pub(crate) type GithubResult<T> = Result<T, GithubError>;

// The user agent for all queries.
pub(crate) const USER_AGENT: &str =
    concat!(env!("CARGO_PKG_NAME"), " v", env!("CARGO_PKG_VERSION"));

/// A client for communicating with a Github instance.
#[derive(Clone)]
pub struct Github {
    /// The client used to communicate with Github.
    client: Client,
    /// The endpoint for REST queries.
    rest_endpoint: Url,
    /// The endpoint for GraphQL queries.
    gql_endpoint: Url,

    /// The authorization process for the client.
    authorization: GithubAuthorization,
}

impl Github {
    fn new_impl(host: &str, authorization: GithubAuthorization) -> GithubResult<Self> {
        let rest_endpoint = Url::parse(&format!("https://{host}/"))?;
        let gql_endpoint = Url::parse(&format!("https://{host}/graphql"))?;

        Ok(Github {
            client: Client::new(),
            rest_endpoint,
            gql_endpoint,
            authorization,
        })
    }

    /// Create a new Github client as a GitHub App.
    ///
    /// The `host` parameter is the API endpoint. For example `github.com` uses `api.github.com`.
    ///
    /// The `app_id` and `private_key` are provided when [registering the application][new-app].
    /// The `installation_id` is an ID associated with a given installation of the application. Its
    /// value is present in webhooks, but does not seem to be available generically.
    ///
    /// [new-app]: https://developer.github.com/apps/building-your-first-github-app/#register-a-new-app-with-github
    pub fn new_app<H, P, I, S>(
        host: H,
        app_id: i64,
        private_key: P,
        installation_ids: I,
    ) -> GithubResult<Self>
    where
        H: AsRef<str>,
        P: AsRef<[u8]>,
        I: IntoIterator<Item = (S, i64)>,
        S: Into<String>,
    {
        let ids = installation_ids
            .into_iter()
            .map(|(s, i)| (s.into(), i))
            .collect();
        let authorization =
            GithubAuthorization::new_app(host.as_ref(), app_id, private_key.as_ref(), ids)?;

        Self::new_impl(host.as_ref(), authorization)
    }

    /// Create a new Github client as a GitHub Action.
    ///
    /// The `host` parameter is the API endpoint. For example `github.com` uses `api.github.com`.
    ///
    /// The `app_id` and `private_key` are provided when [registering the application][new-app].
    /// The `installation_id` is an ID associated with a given installation of the application. Its
    /// value is present in webhooks, but does not seem to be available generically.
    ///
    /// [new-app]: https://developer.github.com/apps/building-your-first-github-app/#register-a-new-app-with-github
    pub fn new_action<H>(host: H) -> GithubResult<Self>
    where
        H: AsRef<str>,
    {
        let authorization = GithubAuthorization::new_action()?;

        Self::new_impl(host.as_ref(), authorization)
    }

    pub(crate) fn app_id(&self) -> Option<i64> {
        self.authorization.app_id()
    }

    pub(crate) fn current_user(&self) -> GithubResult<CurrentUser> {
        self.authorization.current_user(&self.client)
    }

    /// Initialize installations.
    pub fn init_installations(&self, action: InstallationAction) -> GithubResult<()> {
        self.authorization.init_installations(&self.client, action)
    }

    /// Handle an installation event.
    pub fn handle_installation_event(
        &self,
        installation_id: i64,
        behavior: InstallationBehavior,
    ) -> GithubResult<InstallationAction> {
        self.authorization
            .handle_installation_event(&self.client, installation_id, behavior)
    }

    /// The authorization header for GraphQL.
    fn installation_auth_header(&self, owner: &str) -> GithubResult<HeaderMap> {
        let token = self.authorization.token(&self.client, owner)?;
        let mut header_value: HeaderValue = format!("token {token}").parse().unwrap();
        header_value.set_sensitive(true);
        Ok([(header::AUTHORIZATION, header_value)]
            .iter()
            .cloned()
            .collect())
    }

    /// Accept headers for REST.
    fn rest_accept_headers() -> HeaderMap {
        [
            // GitHub v3 API
            (
                header::ACCEPT,
                "application/vnd.github.v3+json".parse().unwrap(),
            ),
        ]
        .iter()
        .cloned()
        .collect()
    }

    /// Accept headers for GraphQL.
    ///
    /// We're using preview APIs and we need these to get access to them.
    fn gql_accept_headers() -> HeaderMap {
        HeaderMap::new()
    }

    pub(crate) fn post<D>(&self, owner: &str, endpoint: &str, data: &D) -> GithubResult<Value>
    where
        D: Serialize,
    {
        let endpoint = Url::parse(&format!("{}{endpoint}", self.rest_endpoint))?;
        let rsp = self
            .client
            .post(endpoint.clone())
            .headers(self.installation_auth_header(owner)?)
            .headers(Self::rest_accept_headers())
            .header(header::USER_AGENT, USER_AGENT)
            .json(data)
            .send()
            .map_err(|err| GithubError::send_request(endpoint, err))?;
        if !rsp.status().is_success() {
            let err = rsp
                .text()
                .unwrap_or_else(|text_err| format!("failed to extract error body: {text_err:?}"));
            return Err(GithubError::github(err));
        }

        rsp.json().map_err(GithubError::json_response)
    }

    /// Send a GraphQL query.
    fn send_impl<Q>(
        &self,
        owner: &str,
        query: &QueryBody<Q::Variables>,
    ) -> GithubResult<Q::ResponseData>
    where
        Q: GraphQLQuery,
        Q::Variables: Debug,
        for<'d> Q::ResponseData: Deserialize<'d>,
    {
        info!(
            target: "github",
            "sending GraphQL query '{}' {:?}",
            query.operation_name,
            query.variables,
        );
        let rsp = self
            .client
            .post(self.gql_endpoint.clone())
            .headers(self.installation_auth_header(owner)?)
            .headers(Self::gql_accept_headers())
            .header(header::USER_AGENT, USER_AGENT)
            .json(query)
            .send()
            .map_err(|err| GithubError::send_request(self.gql_endpoint.clone(), err))?;
        if rsp.status().is_server_error() {
            warn!(
                target: "github",
                "service error {} for query; retrying with backoff",
                rsp.status().as_u16(),
            );
            return Err(GithubError::github_service(rsp.status()));
        }
        if !rsp.status().is_success() {
            let err = rsp
                .text()
                .unwrap_or_else(|text_err| format!("failed to extract error body: {text_err:?}"));
            return Err(GithubError::github(err));
        }

        let rsp: Response<Q::ResponseData> = rsp.json().map_err(GithubError::json_response)?;
        if let Some(errs) = rsp.errors {
            return Err(GithubError::graphql(errs));
        }
        rsp.data.ok_or_else(GithubError::no_response)
    }

    /// Send a GraphQL query.
    pub fn send<Q>(
        &self,
        owner: &str,
        query: &QueryBody<Q::Variables>,
    ) -> GithubResult<Q::ResponseData>
    where
        Q: GraphQLQuery,
        Q::Variables: Debug,
        for<'d> Q::ResponseData: Deserialize<'d>,
    {
        retry_with_backoff(|| self.send_impl::<Q>(owner, query))
    }
}

fn retry_with_backoff<F, K>(mut tryf: F) -> GithubResult<K>
where
    F: FnMut() -> GithubResult<K>,
{
    iter::repeat_n((), BACKOFF_LIMIT)
        .scan(BACKOFF_START, |timeout, _| {
            match tryf() {
                Ok(r) => Some(Some(Ok(r))),
                Err(err) => {
                    if err.should_backoff() {
                        thread::sleep(*timeout);
                        *timeout *= BACKOFF_SCALE;
                        Some(None)
                    } else {
                        Some(Some(Err(err)))
                    }
                },
            }
        })
        .flatten()
        .next()
        .unwrap_or_else(|| Err(GithubError::github_backoff()))
}

#[cfg(test)]
mod tests {
    use reqwest::{header, Client, StatusCode};

    use crate::client::{retry_with_backoff, Github, GithubError, BACKOFF_LIMIT};

    #[test]
    fn test_rest_accept_headers() {
        let rest_headers = Github::rest_accept_headers();
        assert_eq!(rest_headers.len(), 1);
        assert_eq!(
            rest_headers.get(header::ACCEPT).unwrap(),
            "application/vnd.github.v3+json",
        );
    }

    #[test]
    fn test_gql_accept_headers() {
        let gql_headers = Github::gql_accept_headers();
        assert!(gql_headers.is_empty());
    }

    #[test]
    fn test_retry_with_backoff_first_success() {
        let mut call_count = 0;
        retry_with_backoff(|| {
            call_count += 1;
            Ok(())
        })
        .unwrap();
        assert_eq!(call_count, 1);
    }

    #[test]
    fn test_retry_with_backoff_second_success() {
        let mut call_count = 0;
        let mut did_err = false;
        retry_with_backoff(|| {
            call_count += 1;
            if did_err {
                Ok(())
            } else {
                did_err = true;
                Err(GithubError::github_service(
                    StatusCode::INTERNAL_SERVER_ERROR,
                ))
            }
        })
        .unwrap();
        assert_eq!(call_count, 2);
    }

    #[test]
    fn test_retry_with_backoff_no_success() {
        let mut call_count = 0;
        let err = retry_with_backoff::<_, ()>(|| {
            call_count += 1;
            Err(GithubError::github_service(
                StatusCode::INTERNAL_SERVER_ERROR,
            ))
        })
        .unwrap_err();
        assert_eq!(call_count, BACKOFF_LIMIT);
        if let GithubError::GithubBackoff {} = err {
        } else {
            panic!("unexpected error: {}", err);
        }
    }

    #[test]
    fn test_rest_headers_work() {
        let req = Client::new()
            .post("https://nowhere")
            .headers(Github::rest_accept_headers())
            .build()
            .unwrap();

        let headers = req.headers();

        for (key, value) in Github::rest_accept_headers().iter() {
            if !headers.get_all(key).iter().any(|av| av == value) {
                panic!("REST request is missing HTTP header `{}: {:?}`", key, value);
            }
        }
    }

    #[test]
    fn test_graphql_headers_work() {
        let req = Client::new()
            .post("https://nowhere")
            .headers(Github::gql_accept_headers())
            .build()
            .unwrap();

        let headers = req.headers();

        for (key, value) in Github::gql_accept_headers().iter() {
            if !headers.get_all(key).iter().any(|av| av == value) {
                panic!(
                    "GraphQL request is missing HTTP header `{}: {:?}`",
                    key, value,
                );
            }
        }
    }
}
