// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::collections::BTreeMap;
use std::env;
use std::sync::RwLock;
use std::time;

use chrono::{DateTime, Duration, Utc};
use jsonwebtoken::{Algorithm, EncodingKey, Header};
use log::{error, info, warn};
use reqwest::blocking::Client;
use reqwest::header::{self, HeaderMap, HeaderValue};
use reqwest::{Method, Url};
use serde::de::DeserializeOwned;
use serde::{Deserialize, Serialize};
use thiserror::Error;
use ttl_cache::TtlCache;

use crate::client::{GithubError, GithubResult, USER_AGENT};

const LOCK_POISONED: &str = "token lock poisoned";
const TOKEN_SLACK_PERIOD: time::Duration = time::Duration::from_secs(5 * 60);

/// The return type for installation token generation.
#[derive(Deserialize)]
struct InstallationToken {
    token: String,
    expires_at: DateTime<Utc>,
}

#[derive(Debug, Serialize)]
struct Claims {
    iat: i64,
    exp: i64,
    iss: i64,
}

#[derive(Debug, Deserialize)]
pub(crate) struct CurrentUser {
    pub(crate) email: String,
    pub(crate) name: String,
    pub(crate) login: String,
}

#[derive(Debug, Deserialize)]
struct GithubAppUser {
    login: String,
}

#[derive(Debug, Deserialize)]
struct GithubInstallation {
    account: GithubAppUser,
    suspended_at: Option<DateTime<Utc>>,
    suspended_by: Option<GithubAppUser>,
}

/// How to handle installation events.
#[derive(Debug, Clone, Copy)]
pub enum InstallationBehavior {
    /// Suspend unrecognized installations and unsuspend others.
    Suspension,
    /// Delete unrecognized installations.
    Deletion,
    /// Observe installations.
    Observe,
}

impl InstallationBehavior {
    fn action_for_user(
        self,
        user: Option<String>,
        installation_id: i64,
    ) -> PendingInstallationAction {
        PendingInstallationAction {
            action: if user.is_some() {
                self.known_installation_action()
            } else {
                self.unknown_installation_action()
            },
            user,
            installation_id,
        }
    }

    fn known_installation_action(self) -> InstallationAction {
        match self {
            Self::Suspension => InstallationAction::Unsuspended,
            Self::Deletion | Self::Observe => InstallationAction::Ignored,
        }
    }

    fn unknown_installation_action(self) -> InstallationAction {
        match self {
            Self::Suspension => InstallationAction::Suspended,
            Self::Deletion => InstallationAction::Deleted,
            Self::Observe => InstallationAction::Ignored,
        }
    }
}

/// The action taken for an installation.
#[derive(Debug, Clone, Copy)]
pub enum InstallationAction {
    Suspended,
    Unsuspended,
    Deleted,
    Ignored,
}

struct PendingInstallationAction {
    action: InstallationAction,
    user: Option<String>,
    installation_id: i64,
}

#[derive(Debug, Deserialize)]
struct GithubApp {
    // This field isn't used directly, but is useful from the `impl Debug`
    #[allow(dead_code)]
    id: u64,
    name: String,
    owner: GithubAppUser,
}

#[derive(Debug, Error)]
#[non_exhaustive]
pub enum GithubAuthError {
    #[error("jwt error: {}", source)]
    Jwt {
        #[source]
        source: jsonwebtoken::errors::Error,
    },
    #[error("missing installation id: {}", owner)]
    MissingInstallationId { owner: String },
    #[error("key error: {}", source)]
    KeyError {
        #[from]
        source: jsonwebtoken::errors::Error,
    },
    #[error("unknown key format; please file an issue")]
    UnknownKeyFormat {},
}

impl GithubAuthError {
    fn jwt(source: jsonwebtoken::errors::Error) -> Self {
        GithubAuthError::Jwt {
            source,
        }
    }

    fn missing_installation_id(owner: String) -> Self {
        GithubAuthError::MissingInstallationId {
            owner,
        }
    }

    fn unknown_key_format() -> Self {
        Self::UnknownKeyFormat {}
    }
}

type GithubAuthResult<T> = Result<T, GithubAuthError>;

pub(crate) struct GithubAppAuth {
    /// The application ID.
    app_id: i64,
    /// The application private key.
    private_key: EncodingKey,

    /// The base endpoint for REST queries.
    app_endpoint: Url,
    /// The installation ID for each repository owner.
    installation_ids: BTreeMap<String, i64>,
    /// Per-installation tokens for querying GraphQL endpoints.
    tokens: RwLock<TtlCache<String, String>>,
}

impl GithubAppAuth {
    fn jwt(&self) -> GithubAuthResult<String> {
        let header = Header::new(Algorithm::RS256);
        let now = Utc::now();
        let expiration = now + Duration::try_minutes(10).expect("parsing literal int as minutes");
        let claims = Claims {
            iat: now.timestamp(),
            exp: expiration.timestamp(),
            iss: self.app_id,
        };
        jsonwebtoken::encode(&header, &claims, &self.private_key).map_err(GithubAuthError::jwt)
    }

    /// Accept headers for the application.
    ///
    /// The application endpoint is a v3 API.
    fn app_accept_headers(&self) -> HeaderMap {
        [
            // GitHub v3 API
            (
                header::ACCEPT,
                HeaderValue::from_static("application/vnd.github.v3+json"),
            ),
        ]
        .iter()
        .cloned()
        .collect()
    }

    /// Perform a REST query for an application endpoint.
    fn rest_app<T>(&self, client: &Client, method: Method, endpoint: Url) -> GithubResult<T>
    where
        T: DeserializeOwned,
    {
        let rsp = client
            .request(method, endpoint.clone())
            .bearer_auth(self.jwt()?)
            .headers(self.app_accept_headers())
            .header(header::USER_AGENT, USER_AGENT)
            .send()
            .map_err(|err| GithubError::send_request(endpoint, err))?;
        if !rsp.status().is_success() {
            let err = rsp
                .text()
                .unwrap_or_else(|text_err| format!("failed to extract error body: {text_err:?}"));
            return Err(GithubError::github(err));
        }

        rsp.json().map_err(GithubError::json_response)
    }

    /// Fetch the application information.
    fn app(&self, client: &Client) -> GithubResult<GithubApp> {
        let endpoint = self.app_endpoint.join("app")?;
        self.rest_app(client, Method::GET, endpoint)
    }

    /// Fetch a token for our installation.
    ///
    /// GraphQL requires an installation token in order to access some endpoints. This allows us to
    /// do things which only "bots" are allowed to do (such as submitting status runs for pull
    /// requests).
    fn new_installation_token(
        &self,
        client: &Client,
        owner: &str,
    ) -> GithubResult<(String, time::Duration)> {
        let iid = self
            .installation_ids
            .get(owner)
            .ok_or_else(|| GithubAuthError::missing_installation_id(owner.into()))?;
        let endpoint = self
            .app_endpoint
            .join(&format!("app/installations/{iid}/access_tokens"))?;

        let rsp: InstallationToken = self.rest_app(client, Method::POST, endpoint)?;
        // How log GitHub lets the token live for...
        let now = Utc::now();
        let token_duration = rsp
            .expires_at
            .signed_duration_since(now)
            .to_std()
            .map_err(|_| {
                let msg = format!(
                    "GitHub gave us an expiration time in the past: {} (it is now {now})",
                    rsp.expires_at,
                );
                error!(target: "github", "{msg}");
                GithubError::github(msg)
            })?;
        // ...but let's take some time off of it to give us some breathing room.
        let token_duration = token_duration
            .checked_sub(TOKEN_SLACK_PERIOD)
            // Though if that's more time than we have, let's use it while we can.
            .unwrap_or(token_duration);
        Ok((rsp.token, token_duration))
    }

    /// Get the token for GraphQL queries.
    ///
    /// This token can expire, but if it has, it will be refreshed automatically.
    fn token(&self, client: &Client, owner: &str) -> GithubResult<String> {
        // Check for a valid token.
        {
            let lock = self.tokens.read().expect(LOCK_POISONED);
            if let Some(token) = lock.get(owner) {
                return Ok(token.into());
            }
        }

        // No valid token, let's try and get a new one.
        self.refresh_installation_token(client, owner)
    }

    /// Refresh the installation token.
    fn refresh_installation_token(&self, client: &Client, owner: &str) -> GithubResult<String> {
        // Grab an exclusive lock.
        let mut lock = self.tokens.write().expect(LOCK_POISONED);

        // Check if the token is valid again. Multiple queries may have noticed an expired token
        // while one request is fulfilling it. If the token is valid again, use it.
        if let Some(token) = lock.get(owner) {
            return Ok(token.into());
        }

        let (new_token, duration) = self.new_installation_token(client, owner)?;
        // Update the token. The `as_ref_or_update` method cannot be used because
        // `new_installation_token` returns a `Result`.
        lock.insert(owner.into(), new_token.clone(), duration);
        // Return the new token.
        Ok(new_token)
    }

    fn handle_installation_event(
        &self,
        installation_id: i64,
        behavior: InstallationBehavior,
    ) -> PendingInstallationAction {
        let user = self
            .installation_ids
            .iter()
            .filter_map(|(user, iid)| {
                if installation_id == *iid {
                    Some(user)
                } else {
                    None
                }
            })
            .next();

        behavior.action_for_user(user.cloned(), installation_id)
    }

    fn perform_installation_action(
        &self,
        client: &Client,
        pending_action: PendingInstallationAction,
    ) -> GithubResult<InstallationAction> {
        let PendingInstallationAction {
            action,
            user,
            installation_id,
        } = pending_action;

        warn!(
            target: "github",
            "performing {action:?} on installation {installation_id} by {user:?}",
        );

        let installation_endpoint = self
            .app_endpoint
            .join(&format!("app/installations/{installation_id}"))?;
        let install: GithubInstallation =
            self.rest_app(client, Method::GET, installation_endpoint)?;
        let install_user = install.account.login;
        let suspended_adj = if install.suspended_at.is_some() {
            "suspended"
        } else {
            "unsuspended"
        };
        info!(
            target: "github",
            "found {suspended_adj} installation {installation_id} by {install_user}",
        );

        let query = match action {
            InstallationAction::Deleted => {
                info!(
                    target: "github",
                    "Deleted: deleting unrecognized installation {installation_id}",
                );
                let endpoint = format!("app/installations/{installation_id}");
                Some((Method::DELETE, endpoint))
            },
            InstallationAction::Suspended => {
                if let Some(suspended_at) = install.suspended_at {
                    let suspend_user = install
                        .suspended_by
                        .as_ref()
                        .map(|u| u.login.as_str())
                        .unwrap_or("<unknown>");
                    info!(
                        target: "github",
                        "Suspended: ignoring installation {installation_id}; suspended at {suspended_at} by {suspend_user}",
                    );
                    None
                } else {
                    info!(
                        target: "github",
                        "Suspended: suspending installation {installation_id}",
                    );
                    let endpoint = format!("app/installations/{installation_id}/suspended");
                    Some((Method::PUT, endpoint))
                }
            },
            InstallationAction::Unsuspended => {
                if let Some(suspended_at) = install.suspended_at {
                    let suspend_user = install
                        .suspended_by
                        .as_ref()
                        .map(|u| u.login.as_str())
                        .unwrap_or("<unknown>");
                    info!(
                        target: "github",
                        "Unsuspended: unsuspending installation {installation_id}; suspended at {suspended_at} by {suspend_user}",
                    );
                    let endpoint = format!("app/installations/{installation_id}/suspended");
                    Some((Method::DELETE, endpoint))
                } else {
                    info!(
                        target: "github",
                        "Unsuspended: ignoring installation {installation_id}",
                    );
                    None
                }
            },
            InstallationAction::Ignored => {
                info!(
                    target: "github",
                    "Ignored: ignoring installation {installation_id}",
                );
                None
            },
        };

        if let Some((method, endpoint)) = query {
            let endpoint = self.app_endpoint.join(&endpoint)?;
            self.rest_app::<()>(client, method, endpoint)?;
        }

        Ok(action)
    }
}

impl Clone for GithubAppAuth {
    fn clone(&self) -> Self {
        Self {
            app_id: self.app_id,
            private_key: self.private_key.clone(),
            app_endpoint: self.app_endpoint.clone(),
            installation_ids: self.installation_ids.clone(),
            tokens: RwLock::new(TtlCache::new(self.installation_ids.len())),
        }
    }
}

#[derive(Clone)]
pub(crate) struct GithubActionAuth {
    /// The token for querying GraphQL endpoints.
    token: String,
}

#[derive(Clone)]
pub(crate) enum GithubAuthorization {
    App(Box<GithubAppAuth>),
    Action(GithubActionAuth),
}

impl GithubAuthorization {
    pub(crate) fn new_app(
        host: &str,
        app_id: i64,
        private_key: &[u8],
        installation_ids: BTreeMap<String, i64>,
    ) -> GithubResult<Self> {
        let app_endpoint = Url::parse(&format!("https://{host}"))?;

        // Try to detect a PEM-encoded key (which is what GitHub provides).
        let private_key = if private_key.starts_with(b"-----BEGIN RSA PRIVATE KEY-----") {
            EncodingKey::from_rsa_pem(private_key).map_err(GithubAuthError::from)?
        } else {
            return Err(GithubAuthError::unknown_key_format().into());
        };

        Ok(GithubAuthorization::App(Box::new(GithubAppAuth {
            app_id,
            app_endpoint,
            private_key,
            // Use a dummy token which is invalid right now. It will automatically be
            // refreshed when necessary.
            tokens: RwLock::new(TtlCache::new(installation_ids.len())),
            installation_ids,
        })))
    }

    pub(crate) fn new_action() -> GithubResult<Self> {
        let token = env::var("GITHUB_TOKEN").map_err(GithubError::invalid_token)?;

        Ok(GithubAuthorization::Action(GithubActionAuth {
            token,
        }))
    }

    pub(crate) fn current_user(&self, client: &Client) -> GithubResult<CurrentUser> {
        match *self {
            GithubAuthorization::App(ref auth) => {
                let app = auth.app(client)?;

                Ok(CurrentUser {
                    // TODO(github-enterprise): What email to use here?
                    email: format!("{}@users.noreply.github.com", app.owner.login),
                    login: app.owner.login,
                    name: app.name,
                })
            },
            GithubAuthorization::Action(_) => {
                let login = env::var("GITHUB_ACTOR").map_err(GithubError::invalid_actor)?;

                Ok(CurrentUser {
                    // TODO(github-enterprise): What email to use here?
                    email: format!("{login}@users.noreply.github.com"),
                    login: login.clone(),
                    name: login,
                })
            },
        }
    }

    pub(crate) fn app_id(&self) -> Option<i64> {
        match *self {
            GithubAuthorization::App(ref auth) => Some(auth.app_id),
            GithubAuthorization::Action(_) => None,
        }
    }

    pub(crate) fn token(&self, client: &Client, owner: &str) -> GithubResult<String> {
        match *self {
            GithubAuthorization::App(ref auth) => auth.token(client, owner),
            GithubAuthorization::Action(ref auth) => Ok(auth.token.clone()),
        }
    }

    pub(crate) fn init_installations(
        &self,
        client: &Client,
        action: InstallationAction,
    ) -> GithubResult<()> {
        match *self {
            GithubAuthorization::App(ref auth) => {
                for (user, iid) in auth.installation_ids.iter() {
                    let pending_action = PendingInstallationAction {
                        action,
                        user: Some(user.clone()),
                        installation_id: *iid,
                    };
                    auth.perform_installation_action(client, pending_action)?;
                }
                Ok(())
            },
            GithubAuthorization::Action(_) => Ok(()),
        }
    }

    pub(crate) fn handle_installation_event(
        &self,
        client: &Client,
        installation_id: i64,
        behavior: InstallationBehavior,
    ) -> GithubResult<InstallationAction> {
        match *self {
            GithubAuthorization::App(ref auth) => {
                let pending_action = auth.handle_installation_event(installation_id, behavior);
                auth.perform_installation_action(client, pending_action)
            },
            GithubAuthorization::Action(_) => Ok(InstallationAction::Ignored),
        }
    }
}
