// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

use std::borrow::Cow;

use clap::builder::{EnumValueParser, PossibleValue};
use clap::{Arg, ArgAction, ValueEnum};
use ghostflow::actions::test::pipelines::{
    self, TestPipelineUser, TestPipelines, TestPipelinesError,
};
use regex::Regex;

use crate::actions::merge_requests::utils;
use crate::actions::merge_requests::{Action, ActionError, Data, Effect, InnerActionError};
use crate::config::io::TestPipelinesActor;

impl InnerActionError for TestPipelinesError {}

pub struct TestPipelinesAction<'a> {
    test_pipelines: &'a TestPipelines,
    actor: &'a TestPipelinesActor,
}

#[derive(Debug, Clone, Copy)]
enum PipelineAction {
    Manual,
    Unsuccessful,
    Failed,
    Completed,
}

impl ValueEnum for PipelineAction {
    fn value_variants<'a>() -> &'a [Self] {
        &[
            Self::Manual,
            Self::Unsuccessful,
            Self::Failed,
            Self::Completed,
        ]
    }

    fn to_possible_value(&self) -> Option<PossibleValue> {
        Some(PossibleValue::new(match self {
            Self::Manual => "manual",
            Self::Unsuccessful => "unsuccessful",
            Self::Failed => "failed",
            Self::Completed => "completed",
        }))
    }
}

impl From<PipelineAction> for pipelines::TestPipelinesAction {
    fn from(action: PipelineAction) -> Self {
        match action {
            PipelineAction::Manual => Self::StartManual,
            PipelineAction::Unsuccessful => Self::RestartUnsuccessful,
            PipelineAction::Failed => Self::RestartFailed,
            PipelineAction::Completed => Self::RestartAll,
        }
    }
}

impl<'a> TestPipelinesAction<'a> {
    pub fn new(test_pipelines: &'a TestPipelines, actor: &'a TestPipelinesActor) -> Self {
        Self {
            test_pipelines,
            actor,
        }
    }
}

impl Action for TestPipelinesAction<'_> {
    fn help(&self, _: &Data) -> Option<Cow<'static, str>> {
        let msg =
            "Triggers the service's CI APIs. It can start or restart CI jobs within the merge \
             request's latest pipeline. It supports the `--action` (or `-a`) flag to select the \
             action to perform. Supported actions are:\n\n  \
             - `manual`: Start jobs awaiting manual action\n  \
             - `unsuccessful`: Restart jobs which completed without success (including skipped or \
               canceled jobs)\n  \
             - `failed`: Restart jobs which completed with failure\n  \
             - `completed`: Restart all completed jobs\n\n\
             Jobs affected by the action may also be filtered. The `--stage` (or `-s`) flag may \
             be used to give the name of a stage to perform the action on. The `--named` (or \
             `-n`) flag may be used multiple times to only affect jobs which match the argument. \
             When given multiple `--named` arguments, a job may match any of the arguments to be \
             affected.";

        Some(msg.into())
    }

    fn perform(&self, arguments: &[&str], data: &Data) -> Vec<Effect> {
        let mut effects = Vec::new();

        let matches = utils::command_app("test")
            .arg(
                Arg::new("ACTION")
                    .long("action")
                    .short('a')
                    .action(ArgAction::Set)
                    .value_parser(EnumValueParser::<PipelineAction>::new()),
            )
            .arg(
                Arg::new("STAGE")
                    .long("stage")
                    .short('s')
                    .action(ArgAction::Set),
            )
            .arg(
                Arg::new("NAMED")
                    .long("named")
                    .short('n')
                    .action(ArgAction::Append)
                    .number_of_values(1),
            )
            // TODO: Support testing merged branches. This requires support for creating new
            // pipelines and attaching them to a merge request.
            // .arg(Arg::with_name("MERGED").long("merged").short("m").takes_value(false))
            .try_get_matches_from(arguments);

        // TODO: Support testing backport branches. As with testing merged branches, this
        // requires support for creating new pipelines and attaching them to a merge request.

        let matches = match matches {
            Ok(matches) => matches,
            Err(err) => {
                effects.push(ActionError::unrecognized_arguments(err).into());
                return effects;
            },
        };

        let mut builder = pipelines::TestPipelinesOptions::builder();
        let mut err_occurred = false;

        let action = matches
            .get_one::<PipelineAction>("ACTION")
            .copied()
            .map(pipelines::TestPipelinesAction::from);
        if let Some(action) = action {
            builder.action(action);
        }

        if let Some(stage) = matches.get_one::<String>("STAGE") {
            builder.stage(stage);
        }

        if let Some(jobs) = matches.get_many::<String>("NAMED") {
            for job in jobs {
                match Regex::new(job) {
                    Ok(re) => {
                        builder.jobs_matching(re);
                    },
                    Err(_) => {
                        err_occurred = true;
                        let msg = format!("failed to compile `{job}` as a regular expression");
                        effects.push(Effect::error(msg));
                    },
                }
            }
        }

        let user = match self.actor {
            TestPipelinesActor::Self_ => None,
            TestPipelinesActor::Author => {
                Some(TestPipelineUser::from(
                    &data.info.merge_request.author.handle,
                ))
            },
            TestPipelinesActor::Requester => {
                Some(TestPipelineUser::from(&data.action_data.cause.who.handle))
            },
            TestPipelinesActor::PipelineOwner => Some(TestPipelineUser::PipelineOwner),
            TestPipelinesActor::JobOwner => Some(TestPipelineUser::JobOwner),
            TestPipelinesActor::User(user) => Some(TestPipelineUser::from(user)),
        };
        if let Some(user) = user {
            builder.user(user);
        }

        if !err_occurred {
            let options = builder.build().unwrap();
            let res = self
                .test_pipelines
                .test_mr(&data.info.merge_request, &options);

            if let Err(err) = res {
                match err {
                    TestPipelinesError::NoPipelinesAvailable => {
                        let msg = "Pipelines seem to be disabled for this merge request.";
                        effects.push(Effect::error(msg));
                    },
                    TestPipelinesError::NoPipelines => {
                        let msg = "No pipelines found for this merge request.";
                        effects.push(Effect::error(msg));
                    },
                    err => {
                        effects.push(ActionError::from(err).into());
                    },
                }
            }
        }

        effects
    }
}

#[cfg(test)]
mod tests {
    use ghostflow::host::PipelineState;
    use serde_json::{json, Value};

    use crate::handlers::test::*;

    const MERGE_REQUEST_TOPIC: &str = "6f781d4d4d85dfd5a609532a26bcc6fd63fcef51";

    fn test_pipelines_config(actor: &str) -> Value {
        json!({
            "ghostflow/example": {
                "branches": {
                    "master": {
                        "test": {
                            "required_access_level": "maintainer",
                            "backend": "pipelines",
                            "config": {
                                "actor": actor,
                            },
                        },
                    },
                },
            },
        })
    }

    fn test_pipelines_config_custom(user: &str) -> Value {
        json!({
            "ghostflow/example": {
                "branches": {
                    "master": {
                        "test": {
                            "required_access_level": "maintainer",
                            "backend": "pipelines",
                            "config": {
                                "actor": {
                                    "user": user,
                                },
                            },
                        },
                    },
                },
            },
        })
    }

    #[test]
    fn test_command_test_pipelines_bad_arguments() {
        let mut service = TestService::new(
            "test_command_test_pipelines_bad_arguments",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::EnablePipelines,
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --unrecognized"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: unexpected \
               argument '--unrecognized' found",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_no_pipelines() {
        let mut service = TestService::new(
            "test_command_test_pipelines_no_pipelines",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::EnablePipelines,
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: Pipelines seem to be disabled for this merge \
               request.",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_no_pipelines_found() {
        let mut service = TestService::new(
            "test_command_test_pipelines_no_pipelines_found",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: No pipelines found for this merge request.",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_a_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_a_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test -a"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--action <ACTION>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_action_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_action_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --action"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--action <ACTION>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_a_invalid_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_a_invalid_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test -a invalid"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: invalid value \
               'invalid' for '--action <ACTION>'",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_action_invalid_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_action_invalid_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --action invalid"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: invalid value \
               'invalid' for '--action <ACTION>'",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_s_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_s_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test -s"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--stage <STAGE>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_stage_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_stage_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --stage"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--stage <STAGE>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_n_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_n_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test -n"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--named <NAMED>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    #[test]
    fn test_command_test_pipelines_named_missing_argument() {
        let mut service = TestService::new(
            "test_command_test_pipelines_named_missing_argument",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: unrecognized arguments: error: a value is \
               required for '--named <NAMED>' but none was supplied",
        );

        service.launch(config, end).unwrap();
    }

    fn test_command_test_pipelines_action_state(
        test: &str,
        action: &str,
        state: PipelineState,
        should_run: bool,
    ) {
        let mut service = TestService::new(
            test,
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(state, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", &format!("Do: test --action {action}")),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        let service = if should_run {
            service.launch(config, end)
        } else {
            service.launch_timeout(config, end)
        };
        service.unwrap()
    }

    #[test]
    fn test_command_test_pipelines_action_manual_state_manual() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_manual_state_manual",
            "manual",
            PipelineState::Manual,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_manual_state_in_progress() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_manual_state_in_progress",
            "manual",
            PipelineState::InProgress,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_manual_state_canceled() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_manual_state_canceled",
            "manual",
            PipelineState::Canceled,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_manual_state_failed() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_manual_state_failed",
            "manual",
            PipelineState::Failed,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_manual_state_success() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_manual_state_success",
            "manual",
            PipelineState::Success,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_unsuccessful_state_manual() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_unsuccessful_state_manual",
            "unsuccessful",
            PipelineState::Manual,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_unsuccessful_state_in_progress() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_unsuccessful_state_in_progress",
            "unsuccessful",
            PipelineState::InProgress,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_unsuccessful_state_canceled() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_unsuccessful_state_canceled",
            "unsuccessful",
            PipelineState::Canceled,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_unsuccessful_state_failed() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_unsuccessful_state_failed",
            "unsuccessful",
            PipelineState::Failed,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_unsuccessful_state_success() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_unsuccessful_state_success",
            "unsuccessful",
            PipelineState::Success,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_failed_state_manual() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_failed_state_manual",
            "failed",
            PipelineState::Manual,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_failed_state_in_progress() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_failed_state_in_progress",
            "failed",
            PipelineState::InProgress,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_failed_state_canceled() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_failed_state_canceled",
            "failed",
            PipelineState::Canceled,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_failed_state_failed() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_failed_state_failed",
            "failed",
            PipelineState::Failed,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_failed_state_success() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_failed_state_success",
            "failed",
            PipelineState::Success,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_completed_state_manual() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_completed_state_manual",
            "completed",
            PipelineState::Manual,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_completed_state_in_progress() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_completed_state_in_progress",
            "completed",
            PipelineState::InProgress,
            false,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_completed_state_canceled() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_completed_state_canceled",
            "completed",
            PipelineState::Canceled,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_completed_state_failed() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_completed_state_failed",
            "completed",
            PipelineState::Failed,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_action_completed_state_success() {
        test_command_test_pipelines_action_state(
            "test_command_test_pipelines_action_completed_state_success",
            "completed",
            PipelineState::Success,
            true,
        )
    }

    #[test]
    fn test_command_test_pipelines_stage_matches() {
        let mut service = TestService::new(
            "test_command_test_pipelines_stage_matches",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, "stage", "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --stage stage"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_stage_no_match() {
        let mut service = TestService::new(
            "test_command_test_pipelines_stage_no_match",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, "stage", "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --stage not-stage"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch_timeout(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_stage_no_match_unknown() {
        let mut service = TestService::new(
            "test_command_test_pipelines_stage_no_match_unknown",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --stage not-stage"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch_timeout(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_name_matches() {
        let mut service = TestService::new(
            "test_command_test_pipelines_name_matches",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named j"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_name_no_match() {
        let mut service = TestService::new(
            "test_command_test_pipelines_name_no_match",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named unknown"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch_timeout(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_name_matches_many() {
        let mut service = TestService::new(
            "test_command_test_pipelines_name_matches_many",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named unknown --named j"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_name_invalid_regex_comment() {
        let mut service = TestService::new(
            "test_command_test_pipelines_name_invalid_regex_comment",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named ["),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: failed to compile `[` as a regular expression",
        );

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_name_invalid_regex_no_run() {
        let mut service = TestService::new(
            "test_command_test_pipelines_name_invalid_regex_no_run",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test --named ["),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("self");
        let end = EndSignal::trigger_job("ghostflow/example", "job", None);

        service.launch_timeout(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_author() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_author",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("author");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "fork");

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_requester() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_requester",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("requester");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "ghostflow");

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_pipeline_owner() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_pipeline_owner",
            [
                Action::create_user("fork"),
                Action::create_user("pipeline_owner"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_owned_pipeline(
                    "ghostflow",
                    "example",
                    MERGE_REQUEST_TOPIC,
                    "pipeline_owner",
                    false,
                ),
                Action::create_owned_job(PipelineState::Manual, None, "job", "fork"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("pipeline_owner");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "pipeline_owner");

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_job_owner() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_job_owner",
            [
                Action::create_user("fork"),
                Action::create_user("pipeline_owner"),
                Action::create_user("job_owner"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_owned_pipeline(
                    "ghostflow",
                    "example",
                    MERGE_REQUEST_TOPIC,
                    "pipeline_owner",
                    false,
                ),
                Action::create_owned_job(PipelineState::Manual, None, "job", "job_owner"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("job_owner");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "job_owner");

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_custom() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_custom",
            [
                Action::create_user("builder"),
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config_custom("builder");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "builder");

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_actor_custom_unknown() {
        let mut service = TestService::new(
            "test_command_test_pipelines_actor_custom_unknown",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config_custom("builder");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: hosting service error: host error: missing \
               user by name: builder",
        );

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_job_archived() {
        let mut service = TestService::new(
            "test_command_test_pipelines_job_archived",
            [
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_archived_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config("requester");
        let end = EndSignal::mr_comment(
            "Errors:\n\n  \
             - While processing the `test` command: hosting service error: host error: job `job` \
               is archived and may not be started",
        );

        service.launch(config, end).unwrap()
    }

    #[test]
    fn test_command_test_pipelines_multiple() {
        let mut service = TestService::new(
            "test_command_test_pipelines_multiple",
            [
                Action::create_user("builder"),
                Action::create_user("fork"),
                Action::new_project("ghostflow", "example"),
                Action::EnablePipelines,
                Action::create_pipeline("ghostflow", "example", MERGE_REQUEST_TOPIC, false),
                Action::create_job(PipelineState::Manual, None, "job"),
                Action::fork_project("ghostflow", "example", "fork"),
                Action::push("fork", "example", "mr-source", MERGE_REQUEST_TOPIC),
                Action::create_mr("ghostflow", "example", "fork", "", false),
                Action::mr_comment("ghostflow", "Do: test\nDo: test -s stage"),
            ],
        )
        .unwrap();
        let config = test_pipelines_config_custom("builder");
        let end = EndSignal::trigger_job("ghostflow/example", "job", "builder");

        service.launch(config, end).unwrap()
    }
}
