/*ckwg +29
 * Copyright 2013-2018 by Kitware, Inc.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  * Neither name of Kitware, Inc. nor the names of any contributors may be used
 *    to endorse or promote products derived from this software without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * \file
 * \brief core descriptor interface and template implementations
 */

#ifndef VITAL_DESCRIPTOR_H_
#define VITAL_DESCRIPTOR_H_

#include <iostream>
#include <limits>
#include <memory>
#include <vector>

#include <cstring>

/// Shared pointer for base descriptor type
class descriptor;
typedef std::shared_ptr< descriptor > descriptor_sptr;

// a short name for unsigned char
typedef unsigned char byte;

// ------------------------------------------------------------------
/// A representation of a feature descriptor used in matching.
class descriptor
{
public:
  int foo() { return 5; }
  /// Destructor
  virtual ~descriptor() = default;

  virtual descriptor_sptr clone() const = 0;

  /// Access the type info of the underlying data (double or float)
  virtual std::type_info const& data_type() const = 0;

  /// The number of elements of the underlying type
  virtual std::size_t size() const = 0;

  /// The number of bytes used to represent the data
  virtual std::size_t num_bytes() const = 0;

  /// Return the descriptor as pointer to bytes
  /**
   * Subclasses should ensure this always works by storing the data
   * as a continuous byte array.
   * Note that as_bytes returns a pointer to the underlying data while
   * as_double returns a vector of doubles which will be copied from
   * the underlying data if possible.  As_bytes is written this way
   * for speed (no copying) at the cost of being restrictive on sub-classes
   * in terms of the way they lay out their descriptors in memory.
   */
  virtual const byte* as_bytes() const = 0;

  /// Return the descriptor as a vector of doubles
  /**
   * Return an empty vector if this makes no sense
   * for the underlying type.
   */
  virtual std::vector< double > as_double() const = 0;

  /// Equality operator
  bool operator==( descriptor const& other ) const
  {
    if( this->data_type() != other.data_type() ||
        this->size() != other.size() )
    {
      return false;
    }
    auto b1 = this->as_bytes();
    auto b2 = other.as_bytes();

    return std::equal(b1, b1 + this->num_bytes(), b2);
  }

  /// Inequality operator
  bool operator!=( descriptor const& other ) const
  {
    return ! operator==(other);
  }

  /// Returns the node_id for the descriptor.
  /**
   * The node_id is generally the vocabulary tree leaf index computed when
   * the descriptor is quantized in the tree.  Two features with the same
   * node_id are expected to have similar visual appearance.
  */
  virtual unsigned int node_id() const { return 0; }

  /// Sets the node_id for the descriptor.
  /**
   * By default this returns false because this base class has nowhere
   * to store the node_id.  Derived classes that do store the node_id
   * should return true if it successfully stored.
  */
  virtual bool set_node_id(unsigned int node_id) { return false; }

};

// // ------------------------------------------------------------------
// /// output stream operator for a feature
// std::ostream& operator<<( std::ostream& s, const descriptor& d );

// /// input stream operator for a feature
// std::istream& operator>>( std::istream& s, descriptor& d );


#endif // VITAL_DESCRIPTOR_H_
