// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#ifndef vtkWebGPUComputePointCloudMapper_h
#define vtkWebGPUComputePointCloudMapper_h

#include "vtkCallbackCommand.h" // for listening on camera & polydata changes
#include "vtkPolyData.h"        // for the poyldata that is going to be rendered
#include "vtkPolyDataMapper.h"
#include "vtkRenderingWebGPUModule.h" // for the module export macro
#include "vtkSmartPointer.h"          // for smart pointers
#include "vtkWebGPURenderer.h" // for the renderer that this point cloud renderer is going to render into

VTK_ABI_NAMESPACE_BEGIN

/**
 * The point cloud renderer uses WebGPU compute shaders to render the point cells of a polydata onto
 * the framebuffer of a given WebGPURenderer.
 * The renderer support point colors. Colors must then be provided as 4 components tuples. The type
 * does not matter as it will be converted to float32 for the shader
 *
 * This implementation will run into issues if WebGPU uses the OpenGL backend.
 */
class VTKRENDERINGWEBGPU_EXPORT vtkWebGPUComputePointCloudMapper : public vtkPolyDataMapper
{
public:
  vtkTypeMacro(vtkWebGPUComputePointCloudMapper, vtkPolyDataMapper);
  static vtkWebGPUComputePointCloudMapper* New();

  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Renders the given actor with the given renderer.
   *
   * @warning: In its current state, the vtkWebGPUComputePointCloudMapper does not support rendering
   * the actors of two different renderers. This means that calling RenderPiece() once with a first
   * vtkRenderer and then calling RenderPiece() again with another vtkRenderer will yield incorrect
   * results. Two mappers must be used in that case
   */
  void RenderPiece(vtkRenderer* ren, vtkActor* act) override;

protected:
  /**
   * Creates the compute passes and sets up the observers
   */
  vtkWebGPUComputePointCloudMapper();
  ~vtkWebGPUComputePointCloudMapper();
  vtkWebGPUComputePointCloudMapper(const vtkWebGPUComputePointCloudMapper&) = delete;
  void operator=(const vtkWebGPUComputePointCloudMapper&) = delete;

private:
  /**
   * Returns the WebGPU Render window of the given renderer. Nullptr if something goes wrong
   */
  vtkWebGPURenderWindow* GetRendererRenderWindow(vtkRenderer* ren);

  /**
   * Creates the compute pipelione and sets up the compute passes for rendering point clouds
   */
  void Initialize(vtkRenderer* ren);

  /**
   * Copies the depth buffer that contins the depth of the pint sback to the depth buffer of the
   * render window
   */
  void UpdateRenderWindowDepthBuffer(vtkRenderer* ren);

  /**
   * Creates the render pipeline for copying the point depth buffer to the render window's depth
   * buffer using a fragment shader
   */
  void CreateCopyDepthBufferRenderPipeline(vtkWebGPURenderWindow* wgpuRenderWindow);

  /**
   * Dispatches the render pass that copies the point depth buffer to the depth buffer of the render
   * window
   */
  void CopyDepthBufferToRenderWindow(vtkWebGPURenderWindow* wgpuRenderWindow);

  /**
   * Udates various attributes of this mapper if necessary.
   *
   * One example is the size of the point depth buffer used: if the RenderWindow of the given
   * renderer isn't the same size as the size of the current point depth buffer, the point depth
   * buffer will be resized
   */
  void Update(vtkRenderer* ren);

  /**
   * Sets the device of the render window of the given renderer on the compute pipeline
   */
  void UseRenderWindowDevice(vtkRenderer* ren);

  /**
   * Resizes the textures used by the point cloud mapper to the size of the render window
   */
  void ResizeToRenderWindow(vtkRenderer* ren);

  /**
   * Sets up the compute pass that copies the depth buffer of the render window
   */
  void InitializeDepthCopyPass(vtkRenderer* ren);

  /**
   * Sets up the compute pass that renders the points
   */
  void InitializePointRenderPass(vtkRenderer* ren);

  /**
   * Resizes and uploads the points data to be render from the current CachedInput.
   *
   * This function also reconfigures the render compute pass so it uses enough workgroups to cover
   * all the points
   */
  void UploadPointsToGPU();

  /**
   * Resizes and uploads the point colors from the current CachedInput.
   *
   * The only color format supported are point scalars with an unsigned char format and 4
   * components. If the point scalars of the given poyldata do not respect that format, no colors
   * will be uploaded
   */
  void UploadColorsToGPU();

  /**
   * Updates the view projection matrix buffer with the view projection matrix data of the matrix of
   * the WebGPURenderer this compute point cloud renderer is rendering to
   */
  void UploadCameraVPMatrix(vtkRenderer* ren);

  /**
   * Called in GetBounds(). When this method is called, the consider the input
   * to be updated depending on whether this->Static is set or not. This method
   * simply obtains the bounds from the data-object and returns it.
   */
  void ComputeBounds();

  // Whether or not the compute pipeline has been initialized or not
  bool Initialized = false;

  // Compute pipeline for the point cloud rendering
  vtkSmartPointer<vtkWebGPUComputePipeline> ComputePipeline;

  // Compute pass that copies the depth buffer of the render window into the custom depth
  // buffer for rendering the points
  vtkSmartPointer<vtkWebGPUComputePass> CopyDepthPass;
  // Compute pass that renders the points to the framebuffer of the render window of the
  // WebGPURenderer
  vtkSmartPointer<vtkWebGPUComputePass> RenderPointsPass;

  // Custom depth buffer for the render of the points
  vtkSmartPointer<vtkWebGPUComputeBuffer> PointDepthBuffer;
  // Index of the buffer that contains the point data in the render point pass
  int PointBufferIndex = -1;
  // Index of the buffer that holds the colors of the points in float format in the point render
  // pass
  int PointColorBufferIndex = -1;

  // Custom depth buffer that contains the depth of the points after they've been rendered
  int PointDepthBufferIndex = -1;
  // Index of the view-projection matrix buffer in the render point pass
  int CameraVPBufferIndex = -1;
  // Index of the framebuffer in the render point pass
  int FrameBufferRenderTextureIndex = -1;

  // The renderer culling pass always calls GetBounds() on the mappers. We use this opportunity to
  // cache the polyData input so that we can reuse it later without having to call on the expensive
  // GetInput() function
  vtkPolyData* CachedInput = nullptr;
  // MTime of the last points we uploaded to the GPU
  vtkMTimeType LastPointsMTime = 0;
  // MTime of the last point data (for point colors) we uploaded to the GPU
  vtkMTimeType LastPointDataMTime = 0;

  struct CopyDepthBufferRenderPipeline
  {
    wgpu::BindGroup BindGroup = nullptr;
    wgpu::RenderPipeline Pipeline = nullptr;
    wgpu::Buffer FrambufferWidthBuffer = nullptr;
  };

  CopyDepthBufferRenderPipeline CopyDepthBufferPipeline;
};

VTK_ABI_NAMESPACE_END

#endif
