// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "vtkWebGPUComputePointCloudMapper.h"
#include "PointCloudRendererCopyDepth.h"
#include "PointCloudRendererShader.h"
#include "vtkCamera.h"
#include "vtkCommand.h"
#include "vtkMatrix4x4.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkWebGPUComputeBuffer.h"
#include "vtkWebGPUInternalsBindGroup.h"
#include "vtkWebGPUInternalsBindGroupLayout.h"
#include "vtkWebGPUInternalsBuffer.h"
#include "vtkWebGPUInternalsComputePass.h"
#include "vtkWebGPUInternalsComputePassBufferStorage.h"
#include "vtkWebGPUInternalsPipelineLayout.h"
#include "vtkWebGPUInternalsRenderPassDescriptor.h"
#include "vtkWebGPUInternalsRenderPipelineDescriptor.h"
#include "vtkWebGPUInternalsShaderModule.h"
#include "vtkWebGPURenderWindow.h"
#include "vtkWebGPURenderer.h"

// TODO forward declare classes as in ComputePass
// TODO don't use wgpu headers in public headers

VTK_ABI_NAMESPACE_BEGIN

vtkStandardNewMacro(vtkWebGPUComputePointCloudMapper);

//------------------------------------------------------------------------------
vtkWebGPUComputePointCloudMapper::vtkWebGPUComputePointCloudMapper() {}

//------------------------------------------------------------------------------
vtkWebGPUComputePointCloudMapper::~vtkWebGPUComputePointCloudMapper() = default;

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent.GetNextIndent());

  os << indent << "Compute pipeline: ";
  this->ComputePipeline->PrintSelf(os, indent);

  os << indent << "Copy depth pass: ";
  this->CopyDepthPass->PrintSelf(os, indent);

  os << indent << "Render point pass: ";
  this->RenderPointsPass->PrintSelf(os, indent);

  os << indent << "Point depth buffer: ";
  this->PointDepthBuffer->PrintSelf(os, indent);

  os << indent << "PointBufferIndex: " << this->PointBufferIndex << std::endl;
  os << indent << "PointColorBufferIndex: " << this->PointColorBufferIndex << std::endl;

  os << indent << "PointDepthBufferIndex: " << this->PointDepthBufferIndex << std::endl;
  os << indent << "CameraVPBufferIndex: " << this->CameraVPBufferIndex << std::endl;
  os << indent << "FrameBufferRenderTextureIndex: " << this->FrameBufferRenderTextureIndex
     << std::endl;
}

//------------------------------------------------------------------------------
vtkWebGPURenderWindow* vtkWebGPUComputePointCloudMapper::GetRendererRenderWindow(vtkRenderer* ren)
{
  vtkRenderWindow* renderWindow = ren->GetRenderWindow();
  vtkWebGPURenderWindow* wgpuRenderWindow = vtkWebGPURenderWindow::SafeDownCast(renderWindow);

  if (wgpuRenderWindow == nullptr)
  {
    vtkErrorWithObjectMacro(this,
      "The renderer given in vtkWebGPUComputePointCloudMapper::GetRendererRenderWindow doesn't "
      "belong to a WebGPURenderWindow.");

    return nullptr;
  }

  if (!wgpuRenderWindow->GetInitialized())
  {
    vtkErrorWithObjectMacro(this,
      "The render window of the given renderer in "
      "vtkWebGPUComputePointCloudMapper::GetRendererRenderWindow "
      "hasn't been initialized. Did you forget to call vtkRenderWindow::Initialize()?");

    return nullptr;
  }

  return wgpuRenderWindow;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::RenderPiece(vtkRenderer* ren, vtkActor* act)
{
  this->Initialize(ren);
  this->Update(ren);

  // Updating the camera matrix because we cannot know which renderer (and thus which camera)
  // RenderPiece was called with
  this->UploadCameraVPMatrix(ren);

  auto wgpuRenWin = vtkWebGPURenderWindow::SafeDownCast(ren->GetRenderWindow());
  if (wgpuRenWin->CheckAbortStatus())
  {
    return;
  }

  if (this->CachedInput == nullptr)
  {
    if (!this->Static)
    {
      this->GetInputAlgorithm()->Update();
    }
    this->CachedInput = this->GetInput();
  }

  const auto device = wgpuRenWin->GetDevice();
  auto wgpuActor = reinterpret_cast<vtkWebGPUActor*>(act);

  vtkWebGPUActor::MapperRenderType renderType = wgpuActor->GetMapperRenderType();
  switch (renderType)
  {
    case vtkWebGPUActor::MapperRenderType::UpdateBuffers:
    {
      this->UploadPointsToGPU();
      this->UploadColorsToGPU();

      break;
    }

    case vtkWebGPUActor::MapperRenderType::RenderPassEncode:
    case vtkWebGPUActor::MapperRenderType::RenderBundleEncode:
    {
      vtkWebGPURenderer* wgpuRenderer = vtkWebGPURenderer::SafeDownCast(ren);
      if (wgpuRenderer == nullptr)
      {
        vtkLog(ERROR,
          "The renderer passed in RenderPiece of vtkWebGPUComputePointCloudMapper is not a WebGPU "
          "Renderer.");

        return;
      }

      wgpuRenderer->AddPostRasterizationActor(act);

      break;
    }

    case vtkWebGPUActor::MapperRenderType::RenderPostRasterization:
    {
      this->ComputePipeline->DispatchAllPasses();
      this->ComputePipeline->Update();

      this->UpdateRenderWindowDepthBuffer(ren);

      break;
    }
  }
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::UpdateRenderWindowDepthBuffer(vtkRenderer* ren)
{
  vtkWebGPURenderWindow* wgpuRenderWindow = this->GetRendererRenderWindow(ren);
  if (wgpuRenderWindow == nullptr)
  {
    return;
  }

  if (this->CopyDepthBufferPipeline.Pipeline == nullptr)
  {
    this->CreateCopyDepthBufferRenderPipeline(wgpuRenderWindow);
  }

  this->CopyDepthBufferToRenderWindow(wgpuRenderWindow);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::CreateCopyDepthBufferRenderPipeline(
  vtkWebGPURenderWindow* wgpuRenderWindow)
{
  wgpu::Device device = wgpuRenderWindow->GetDevice();

  // Creating the buffer that will hold the width of the framebuffer for the fragment shader that
  // copies the point depth buffer into the depth buffer of the render window
  this->CopyDepthBufferPipeline.FrambufferWidthBuffer = vtkWebGPUInternalsBuffer::CreateABuffer(
    device, sizeof(unsigned int), wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::Uniform, false,
    "Point cloud mapper - Copy depth to RenderWindow - Framebuffer width uniform buffer");

  wgpu::BindGroupLayout bgl = vtkWebGPUInternalsBindGroupLayout::MakeBindGroupLayout(device,
    {
      // clang-format off
      { 0, wgpu::ShaderStage::Fragment, wgpu::BufferBindingType::ReadOnlyStorage },
      { 1, wgpu::ShaderStage::Fragment, wgpu::BufferBindingType::Uniform }
      // clang-format on
    });
  bgl.SetLabel("FSQ bind group layout");

  wgpu::PipelineLayout pipelineLayout =
    vtkWebGPUInternalsPipelineLayout::MakeBasicPipelineLayout(device, &bgl);
  pipelineLayout.SetLabel("FSQ graphics pipeline layout");

  this->CopyDepthBufferPipeline.BindGroup = vtkWebGPUInternalsBindGroup::MakeBindGroup(device, bgl,
    {
      // clang-format off
      { 0, this->CopyDepthPass->Internals->BufferStorage->GetWGPUBuffer(this->PointDepthBufferIndex) },
      { 1, this->CopyDepthBufferPipeline.FrambufferWidthBuffer }
      // clang-format on
    });

  // TODO put that in a separate file
  wgpu::ShaderModule shaderModule = vtkWebGPUInternalsShaderModule::CreateFromWGSL(device, R"(
    struct VertexOutput {
      @builtin(position) position: vec4<f32>,
      @location(0) uv: vec2<f32>
    }

    @vertex
    fn vertexMain(@builtin(vertex_index) vertex_id: u32) -> VertexOutput {
      var output: VertexOutput;
      var coords: array<vec2<f32>, 4> = array<vec2<f32>, 4>(
        vec2<f32>(-1, -1), // bottom-left
        vec2<f32>(-1,  1), // top-left
        vec2<f32>( 1, -1), // bottom-right
        vec2<f32>( 1,  1)  // top-right
      );

      output.position = vec4<f32>(coords[vertex_id].xy, 1.0, 1.0);
      output.uv = output.position.xy * 0.5 + 0.5;

      // flip y for texture coordinate.

      output.uv.y = 1.0 - output.uv.y;

      return output;
    }

    struct FragmentInput {
      @builtin(position) position: vec4<f32>,
      @location(0) uv: vec2<f32>
    }

    struct FragmentOutput {
      @builtin(frag_depth) fragmentDepth: f32
    }

    @group(0) @binding(0) var<storage, read> pointDepthBuffer: array<u32>;
    @group(0) @binding(1) var<uniform> framebufferWidth: u32;

    @fragment
    fn fragmentMain(fragment: FragmentInput) -> FragmentOutput {
      var fragOut: FragmentOutput;
      
      let pixelIndex = u32(fragment.position.x) + u32(fragment.position.y) * framebufferWidth;
      fragOut.fragmentDepth = f32(pointDepthBuffer[pixelIndex]) / (pow(2.0, 32.0) - 1);

      return fragOut;
    }
  )");

  vtkWebGPUInternalsRenderPipelineDescriptor pipelineDesc;
  pipelineDesc.label = "Point cloud mapper - Copy point depth buffer graphics pipeline description";
  pipelineDesc.layout = pipelineLayout;
  pipelineDesc.vertex.module = shaderModule;
  pipelineDesc.vertex.entryPoint = "vertexMain";
  pipelineDesc.vertex.bufferCount = 0;
  pipelineDesc.cFragment.module = shaderModule;
  pipelineDesc.cFragment.entryPoint = "fragmentMain";
  // We are not going to use the color target but Dawn needs it
  pipelineDesc.cFragment.targetCount = 1;
  pipelineDesc.cTargets[0].format = wgpuRenderWindow->GetPreferredSwapChainTextureFormat();
  // Not writing to the color attachment
  pipelineDesc.cTargets[0].writeMask = wgpu::ColorWriteMask::None;

  // Enabling the depth buffer
  wgpu::DepthStencilState* depthState = pipelineDesc.EnableDepthStencil();
  depthState->depthWriteEnabled = true;
  depthState->depthCompare = wgpu::CompareFunction::Less;
  pipelineDesc.primitive.topology = wgpu::PrimitiveTopology::TriangleStrip;

  this->CopyDepthBufferPipeline.Pipeline = device.CreateRenderPipeline(&pipelineDesc);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::CopyDepthBufferToRenderWindow(
  vtkWebGPURenderWindow* wgpuRenderWindow)
{
  std::vector<wgpu::TextureView> colorAttachment;
  colorAttachment.push_back(wgpuRenderWindow->GetOffscreenColorAttachmentView());

  vtkWebGPUInternalsRenderPassDescriptor renderPassDescriptor(colorAttachment,
    wgpuRenderWindow->GetDepthStencilView(),
    /* Don't clear the color/depth buffer with this pass */ false);
  // Discarding anything that we write to the color attachment
  renderPassDescriptor.ColorAttachments[0].storeOp = wgpu::StoreOp::Store;

  int* windowSize = wgpuRenderWindow->GetSize();

  wgpu::CommandEncoderDescriptor encDesc = {};
  std::stringstream label;
  encDesc.label = "vtkWebGPURenderWindow::CommandEncoder";
  wgpu::CommandEncoder commandEncoder =
    wgpuRenderWindow->GetDevice().CreateCommandEncoder(&encDesc);

  unsigned int framebufferWidth = windowSize[0];
  commandEncoder.WriteBuffer(this->CopyDepthBufferPipeline.FrambufferWidthBuffer, 0,
    (uint8_t*)&framebufferWidth, sizeof(unsigned int));

  auto encoder = commandEncoder.BeginRenderPass(&renderPassDescriptor);
  encoder.SetLabel("Point cloud mapper - Encode copy point depth buffer to render window");
  encoder.SetViewport(0, 0, windowSize[0], windowSize[1], 0.0, 1.0);
  encoder.SetScissorRect(0, 0, windowSize[0], windowSize[1]);
#ifndef NDEBUG
  encoder.PushDebugGroup("Point cloud mapper - Copy point depth buffer to render window");
#endif
  encoder.SetPipeline(this->CopyDepthBufferPipeline.Pipeline);
  encoder.SetBindGroup(0, this->CopyDepthBufferPipeline.BindGroup);
  encoder.Draw(4);
#ifndef NDEBUG
  encoder.PopDebugGroup();
#endif
  encoder.End();

  wgpu::CommandBufferDescriptor cmdBufDesc = {};
  wgpu::CommandBuffer cmdBuffer = commandEncoder.Finish(&cmdBufDesc);
  wgpuRenderWindow->FlushCommandBuffers(1, &cmdBuffer);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::Initialize(vtkRenderer* ren)
{
  if (this->Initialized)
  {
    // Already initialized

    return;
  }

  this->ComputePipeline = vtkSmartPointer<vtkWebGPUComputePipeline>::New();

  this->CopyDepthPass = this->ComputePipeline->CreateComputePass();
  this->CopyDepthPass->SetShaderSource(PointCloudRendererCopyDepth);
  this->CopyDepthPass->SetShaderEntryPoint("computeMain");

  this->RenderPointsPass = this->ComputePipeline->CreateComputePass();
  this->RenderPointsPass->SetShaderSource(PointCloudRendererShader);
  this->RenderPointsPass->SetShaderEntryPoint("pointCloudRenderEntryPoint");

  this->UseRenderWindowDevice(ren);
  this->InitializeDepthCopyPass(ren);
  this->InitializePointRenderPass(ren);

  this->Initialized = true;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::Update(vtkRenderer* ren)
{
  this->ResizeToRenderWindow(ren);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::UseRenderWindowDevice(vtkRenderer* ren)
{
  vtkWebGPURenderWindow* wgpuRenderWindow = this->GetRendererRenderWindow(ren);
  if (wgpuRenderWindow == nullptr)
  {
    return;
  }

  this->ComputePipeline->SetDevice(wgpuRenderWindow->GetDevice());
  this->ComputePipeline->SetAdapter(wgpuRenderWindow->GetAdapter());
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::ResizeToRenderWindow(vtkRenderer* ren)
{
  vtkWebGPURenderWindow* wgpuRenderWindow = this->GetRendererRenderWindow(ren);
  if (wgpuRenderWindow == nullptr)
  {
    return;
  }

  int* windowSize = wgpuRenderWindow->GetSize();

  if (this->CopyDepthPass->GetBufferByteSize(this->PointDepthBufferIndex) ==
    windowSize[0] * windowSize[1] * sizeof(unsigned int))
  {
    // Nothing to resize

    return;
  }

  this->CopyDepthPass->ResizeBuffer(
    this->PointDepthBufferIndex, windowSize[0] * windowSize[1] * sizeof(unsigned int));

  int nbGroupsX = std::ceil(windowSize[0] / 8.0f);
  int nbGroupsY = std::ceil(windowSize[1] / 8.0f);
  this->CopyDepthPass->SetWorkgroups(nbGroupsX, nbGroupsY, 1);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::InitializeDepthCopyPass(vtkRenderer* ren)
{
  vtkWebGPURenderWindow* wgpuRenderWindow = this->GetRendererRenderWindow(ren);
  if (wgpuRenderWindow == nullptr)
  {
    return;
  }

  int* windowSize = wgpuRenderWindow->GetSize();

  vtkSmartPointer<vtkWebGPUComputeRenderTexture> depthBufferRenderTexture;
  depthBufferRenderTexture = wgpuRenderWindow->AcquireDepthBufferRenderTexture();
  int depthBufferRenderTextureIndex =
    this->CopyDepthPass->AddRenderTexture(depthBufferRenderTexture);

  vtkSmartPointer<vtkWebGPUComputeTextureView> depthBufferTextureView =
    this->CopyDepthPass->CreateTextureView(depthBufferRenderTextureIndex);
  depthBufferTextureView->SetGroup(0);
  depthBufferTextureView->SetBinding(0);
  depthBufferTextureView->SetAspect(vtkWebGPUComputeTextureView::TextureViewAspect::ASPECT_DEPTH);
  depthBufferTextureView->SetMode(vtkWebGPUComputeTextureView::TextureViewMode::READ_ONLY);
  depthBufferTextureView->SetFormat(vtkWebGPUComputeTexture::DEPTH_24_PLUS);
  depthBufferTextureView->SetLabel("Point cloud mapper - point render pass - depth buffer");
  this->CopyDepthPass->AddTextureView(depthBufferTextureView);

  this->PointDepthBuffer = vtkSmartPointer<vtkWebGPUComputeBuffer>::New();
  this->PointDepthBuffer->SetGroup(0);
  this->PointDepthBuffer->SetBinding(1);
  this->PointDepthBuffer->SetByteSize(windowSize[0] * windowSize[1] * sizeof(unsigned int));
  this->PointDepthBuffer->SetMode(vtkWebGPUComputeBuffer::BufferMode::READ_WRITE_COMPUTE_STORAGE);
  this->PointDepthBufferIndex = this->CopyDepthPass->AddBuffer(this->PointDepthBuffer);

  int nbGroupsX = std::ceil(windowSize[0] / 8.0f);
  int nbGroupsY = std::ceil(windowSize[1] / 8.0f);
  this->CopyDepthPass->SetWorkgroups(nbGroupsX, nbGroupsY, 1);
  this->CopyDepthPass->SetLabel("Point cloud mapper - Depth buffer copy pass");
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::InitializePointRenderPass(vtkRenderer* ren)
{
  vtkWebGPURenderWindow* wgpuRenderWindow = this->GetRendererRenderWindow(ren);
  if (wgpuRenderWindow == nullptr)
  {
    return;
  }

  auto colorFramebuffer = wgpuRenderWindow->AcquireFramebufferRenderTexture();

  vtkSmartPointer<vtkWebGPUComputeBuffer> pointBuffer;
  pointBuffer = vtkSmartPointer<vtkWebGPUComputeBuffer>::New();
  pointBuffer->SetGroup(0);
  pointBuffer->SetBinding(0);
  // Will be resized when the polydata will be set on this point cloud renderer
  pointBuffer->SetByteSize(1);
  pointBuffer->SetDataType(vtkWebGPUComputeBuffer::BufferDataType::STD_VECTOR);
  pointBuffer->SetLabel("Point cloud mapper - point render pass - point buffer");
  pointBuffer->SetMode(vtkWebGPUComputeBuffer::BufferMode::READ_ONLY_COMPUTE_STORAGE);
  this->PointBufferIndex = this->RenderPointsPass->AddBuffer(pointBuffer);

  // Binding (0, 1)
  this->RenderPointsPass->AddBuffer(this->PointDepthBuffer);

  vtkSmartPointer<vtkWebGPUComputeBuffer> pointColorBuffer;
  pointColorBuffer = vtkSmartPointer<vtkWebGPUComputeBuffer>::New();
  pointColorBuffer->SetGroup(0);
  pointColorBuffer->SetBinding(2);
  pointColorBuffer->SetDataType(vtkWebGPUComputeBuffer::BufferDataType::STD_VECTOR);
  // Dummy size. Will be resized when setting the polydata
  pointColorBuffer->SetByteSize(4);
  pointColorBuffer->SetMode(vtkWebGPUComputeBuffer::BufferMode::READ_ONLY_COMPUTE_STORAGE);
  pointColorBuffer->SetLabel("Point cloud mapper - point render pass - point color buffer");
  this->PointColorBufferIndex = this->RenderPointsPass->AddBuffer(pointColorBuffer);

  int framebufferIndex = this->RenderPointsPass->AddRenderTexture(colorFramebuffer);
  vtkSmartPointer<vtkWebGPUComputeTextureView> framebufferTextureView =
    this->RenderPointsPass->CreateTextureView(framebufferIndex);
  framebufferTextureView->SetGroup(0);
  framebufferTextureView->SetBinding(3);
  framebufferTextureView->SetFormat(vtkWebGPUComputeTexture::TextureFormat::BGRA8_UNORM);
  framebufferTextureView->SetMode(vtkWebGPUComputeTextureView::TextureViewMode::WRITE_ONLY_STORAGE);
  framebufferTextureView->SetAspect(vtkWebGPUComputeTextureView::TextureViewAspect::ASPECT_ALL);
  framebufferTextureView->SetLabel("Point cloud mapper - point render pass - color framebuffer");
  this->RenderPointsPass->AddTextureView(framebufferTextureView);

  vtkSmartPointer<vtkWebGPUComputeBuffer> cameraVPBuffer;
  cameraVPBuffer = vtkSmartPointer<vtkWebGPUComputeBuffer>::New();
  cameraVPBuffer->SetGroup(0);
  cameraVPBuffer->SetBinding(4);
  cameraVPBuffer->SetByteSize(sizeof(float) * 4 * 4); // 4x4 matrix
  cameraVPBuffer->SetLabel(
    "Compute point cloud renderer - Point render pass - Camera view-projection matrix buffer");
  cameraVPBuffer->SetDataType(vtkWebGPUComputeBuffer::BufferDataType::STD_VECTOR);
  cameraVPBuffer->SetMode(vtkWebGPUComputeBuffer::BufferMode::UNIFORM_BUFFER);

  this->CameraVPBufferIndex = this->RenderPointsPass->AddBuffer(cameraVPBuffer);
  this->RenderPointsPass->SetLabel("Point cloud mapper - Render point pass pass");
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::UploadPointsToGPU()
{
  vtkPoints* points = this->CachedInput->GetPoints();
  vtkMTimeType pointsMTime = points->GetMTime();
  if (points->GetMTime() <= this->LastPointsMTime)
  {
    // Nothing to upload, already up to date

    return;
  }

  this->LastPointsMTime = pointsMTime;

  std::vector<float> floatPointsData(points->GetNumberOfPoints() * 3);
  for (vtkIdType i = 0; i < points->GetNumberOfPoints(); i++)
  {
    double* pointDouble = points->GetPoint(i);

    floatPointsData[i * 3 + 0] = static_cast<float>(pointDouble[0]);
    floatPointsData[i * 3 + 1] = static_cast<float>(pointDouble[1]);
    floatPointsData[i * 3 + 2] = static_cast<float>(pointDouble[2]);
  }

  // The compute shader uses workgroups of size (256, 1, 1)
  // The maximum number of workgroups on one dimensions is 65535
  // as per the spec:
  // https://www.w3.org/TR/webgpu/#dom-supported-limits-maxcomputeworkgroupsperdimension
  //
  // If 65535 workgroups of size 256 isn't enough to have one thread per point (that's
  // a maximum of 16776960 thread per compute invocation), then 1 single thread will render
  // multiple points. This logic is handled in the shader.
  int groupsX =
    static_cast<int>(std::min(65535.0f, std::ceil(points->GetNumberOfPoints() / 256.0f)));
  this->RenderPointsPass->SetWorkgroups(groupsX, 1, 1);
  this->RenderPointsPass->ResizeBuffer(
    this->PointBufferIndex, points->GetNumberOfPoints() * sizeof(float) * 3);
  this->RenderPointsPass->UpdateBufferData(this->PointBufferIndex, floatPointsData);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::UploadColorsToGPU()
{
  vtkMTimeType pointDataMTime = this->CachedInput->GetPointData()->GetMTime();
  if (pointDataMTime <= this->LastPointDataMTime)
  {
    // Nothing to upload, already up to date

    return;
  }

  this->LastPointDataMTime = pointDataMTime;

  vtkUnsignedCharArray* pointColors = this->MapScalars(1.0);

  if (pointColors != nullptr && pointColors->GetNumberOfValues() > 0)
  {
    // If we DO have colors to upload

    // Resizing to hold unsigned char type colors (the documentation of MapScalars() guarantees that
    // this->Colors contains unsigned char data)
    this->RenderPointsPass->ResizeBuffer(
      this->PointColorBufferIndex, sizeof(unsigned char) * pointColors->GetNumberOfValues());
    this->RenderPointsPass->UpdateBufferData(this->PointColorBufferIndex, pointColors);
  }
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::UploadCameraVPMatrix(vtkRenderer* ren)
{
  vtkCamera* camera = ren->GetActiveCamera();

  // Resetting the clipping range of the camera with the cached bounds of the polydata being
  // rendered.
  // The bounds may be nullptr if we haven't set the polydata yet
  // ren->ResetCameraClippingRange(this->GetBounds());

  vtkMatrix4x4* viewMatrix = camera->GetModelViewTransformMatrix();
  vtkMatrix4x4* projectionMatrix =
    camera->GetProjectionTransformMatrix(ren->GetTiledAspectRatio(), -1, 1);
  vtkNew<vtkMatrix4x4> viewProj;
  vtkMatrix4x4::Multiply4x4(projectionMatrix, viewMatrix, viewProj);
  // WebGPU uses column major matrices but VTK is row major
  viewProj->Transpose();

  // Getting the matrix data as floats
  std::vector<float> matrixData(16);
  for (int i = 0; i < 16; i++)
  {
    matrixData[i] = static_cast<float>(viewProj->GetData()[i]);
  }

  this->RenderPointsPass->UpdateBufferData(this->CameraVPBufferIndex, matrixData);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePointCloudMapper::ComputeBounds()
{
  // Caching the input so that it can be reused by the function that uploads
  this->CachedInput = this->GetInput();

  this->InvokeEvent(vtkCommand::StartEvent, nullptr);
  if (!this->Static)
  {
    this->GetInputAlgorithm()->Update();
  }
  this->InvokeEvent(vtkCommand::EndEvent, nullptr);
  if (!this->CachedInput)
  {
    vtkMath::UninitializeBounds(this->Bounds);
    return;
  }

  // Only considering the bounds of the points, not the cells
  this->CachedInput->GetPoints()->GetBounds(this->Bounds);
}
