// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "vtkWebGPUComputeTexture.h"
#include "vtkNew.h"           // for the new macro
#include "vtkObjectFactory.h" // for the new macro

#include <vector>

vtkStandardNewMacro(vtkWebGPUComputeTexture);

//------------------------------------------------------------------------------
vtkWebGPUComputeTexture::vtkWebGPUComputeTexture() = default;

//------------------------------------------------------------------------------
vtkWebGPUComputeTexture::~vtkWebGPUComputeTexture() = default;

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::PrintSelf(ostream& os, vtkIndent indent)
{
  os << indent << "ByteSize: " << this->ByteSize << std::endl;
  os << indent << "Extents X/Y/Z: " << this->Extents[0] << ", " << this->Extents[1] << ", "
     << this->Extents[2] << std::endl;

  os << indent << "Dimension: " << this->Dimension << std::endl;
  os << indent << "Format: " << this->Format << std::endl;
  os << indent << "Mode: " << this->Mode << std::endl;
  os << indent << "SampleType: " << this->SampleType << std::endl;
  os << indent << "MipLevelCount: " << this->MipLevelCount << std::endl;
  os << indent << "DataType: " << this->DataType << std::endl;
  os << indent << "DataPointer: " << this->DataPointer << std::endl;
  os << indent << "DataArray: " << this->DataArray << std::endl;
  os << indent << "Label: " << this->Label << std::endl;
}

//------------------------------------------------------------------------------
unsigned int vtkWebGPUComputeTexture::GetBytesPerPixel()
{
  switch (this->Format)
  {
    case vtkWebGPUComputeTexture::TextureFormat::RGBA8_UNORM:
      return 4;

    case vtkWebGPUComputeTexture::TextureFormat::R32_FLOAT:
      return 4;

    default:
      vtkLog(ERROR,
        "Unhandled texture format in vtkWebGPUComputeTexture::GetBytesPerPixel: " << this->Format);
  }

  return 0;
}

//------------------------------------------------------------------------------
unsigned int vtkWebGPUComputeTexture::GetPixelComponentsCount()
{
  switch (this->Format)
  {
    case vtkWebGPUComputeTexture::TextureFormat::RGBA8_UNORM:
      return 4;

    case vtkWebGPUComputeTexture::TextureFormat::R32_FLOAT:
      return 1;

    default:
      vtkLog(ERROR,
        "Unhandled texture format in vtkWebGPUComputeTexture::GetPixelComponentsCount: "
          << this->Format);
  }

  return 0;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::SetData(vtkDataArray* data)
{
  this->DataArray = data;
  this->ByteSize = data->GetNumberOfValues() * data->GetDataTypeSize();
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::GetSize(unsigned int& x, unsigned int& y, unsigned int& z)
{
  x = this->Extents[0];
  y = this->Extents[1];
  z = this->Extents[2];
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::GetSize(unsigned int& x, unsigned int& y)
{
  x = this->Extents[0];
  y = this->Extents[1];
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::GetSize(unsigned int* xyz)
{
  xyz = this->Extents;
}

//------------------------------------------------------------------------------
unsigned int* vtkWebGPUComputeTexture::GetSize()
{
  return this->Extents;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::SetSize(unsigned int x, unsigned int y, unsigned int z)
{
  this->Extents[0] = x;
  this->Extents[1] = y;
  this->Extents[2] = z;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::SetSize(unsigned int x, unsigned int y)
{
  this->Extents[0] = x;
  this->Extents[1] = y;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputeTexture::SetSize(unsigned int* xyz)
{
  this->Extents[0] = xyz[0];
  this->Extents[1] = xyz[1];
  this->Extents[2] = xyz[2];
}
